#include <windows.h>
#include "ddraw.h"
#include "etc_func.h"
#include "directdraw.h"

XBuffer::XBuffer()
{
	m_Address = NULL;
	m_AllocateFlag = FALSE;
	m_Width = 0;
	m_Height = 0;
	m_Pitch = 0;
}

XBuffer::XBuffer(int width, int height)
{
	Create(width, height);
}

XBuffer::~XBuffer()
{
	if(m_AllocateFlag == TRUE) 
		delete [] m_Address;
}

int XBuffer::Create(int width, int height)
{
	if(m_AllocateFlag == TRUE) return FALSE;
	m_Address = new char [width * height];
	if(m_Address == NULL) return FALSE;
	m_AllocateFlag = TRUE;
	m_Width = width;
	m_Height = height;
	m_Pitch = width;
	SetClipper(GetBorder());

	return TRUE;
}

int XBuffer::GetWidth()
{
	return m_Width;
}

int XBuffer::GetHeight()
{
	return m_Height;
}

int XBuffer::GetPitch()
{
	return m_Pitch;
}

XRectangle& XBuffer::GetClipper()
{
	return m_Clipper;
}

XRectangle XBuffer::GetBorder()
{
	return XRectangle(0,0,m_Width-1,m_Height-1);
}

char *XBuffer::GetAddress()
{
	return m_Address;
}

void XBuffer::SetWidth(int width)
{
	m_Width = width;
}

void XBuffer::SetHeight(int height)
{
	m_Height = height;
}

void XBuffer::SetPitch(int pitch)
{
	m_Pitch = pitch;
}

void XBuffer::SetAddress(char *address)
{
	m_Address = address;
}

void XBuffer::SetClipper(XRectangle rectangle)
{
	m_Clipper = rectangle;
	m_Clipper.ClipEdge(GetBorder());
}

void XBuffer::SetClipper(int left, int top, int right, int bottom)
{
	SetClipper(XRectangle(left,top,right,bottom));
}

int XBuffer::LoadBitmap(char *name)
{
	BITMAPFILEHEADER bfh;
	BITMAPINFOHEADER bih;

	if(LoadBitmapHeader(name, &bfh, &bih) == FALSE) return FALSE;
	if( m_Width != bih.biWidth || m_Height != bih.biHeight)
	{
		if(m_AllocateFlag == FALSE) return FALSE;
		m_Width = bih.biWidth;
		m_Height = bih.biHeight;
		m_Pitch = m_Width;
		delete [] m_Address; 
		m_AllocateFlag = FALSE; 
		m_Address = new char [m_Width * m_Height];
		if(m_Address == NULL) return FALSE;
		m_AllocateFlag = TRUE;
		SetClipper(GetBorder());
	}

	return LoadBitmapDataForPitch(name, m_Address, m_Pitch);
}

void XBuffer::Clear(char color)
{
	DrawFillRectangle(GetBorder(),color);
}

void XBuffer::DrawPixel(int x, int y, char color)
{
	if(m_Clipper.CheckCrash(x, y) != TRUE) return;

	m_Address[x+y*m_Pitch] = color;
}

void XBuffer::DrawHLine(int left, int right, int y, char color)
{
	if(left > right) Swap(left, right);
	if(right < m_Clipper.left || left > m_Clipper.right) return;
	if(y < m_Clipper.top || y > m_Clipper.bottom) return;
	if(left < m_Clipper.left) left = m_Clipper.left;
	if(right > m_Clipper.right) right = m_Clipper.right;

	int count = right-left+1;
	char *address = m_Address+left+y*m_Pitch;

	__asm
	{
		mov edi, address
		mov ah, color
		mov al, color
		push ax
		push ax
		pop eax
		mov ecx, count
		shr ecx, 1
		jnc goto1
		stosb
goto1:	shr ecx, 1
		jnc goto2
		stosw
goto2:	rep stosd
	}
}

void XBuffer::DrawVLine(int x, int top, int bottom, char color)
{
	if(top > bottom) Swap(top, bottom);
	if(x < m_Clipper.left || x > m_Clipper.right) return;
	if(bottom < m_Clipper.top || top > m_Clipper.bottom) return;
	if(bottom > m_Clipper.bottom) bottom = m_Clipper.bottom;
	if(top < m_Clipper.top) top = m_Clipper.top;

	int count = bottom-top+1;
	int add_width = m_Pitch-1;
	char *address = m_Address+x+top*m_Pitch;

	__asm
	{
		mov edi, address
		mov al, color
		mov ecx, count
goto1:	stosb
		dec ecx
		jz goto2
		add edi, add_width
		jmp goto1
goto2:	
	}
}

void XBuffer::DrawLine(int x1, int y1, int x2, int y2, char color)
{
	if((x1 < m_Clipper.left && x2 < m_Clipper.left)||(x1 > m_Clipper.right && x2 > m_Clipper.right)) return;
	if((y1 < m_Clipper.top && y2 < m_Clipper.top)||(y1 > m_Clipper.bottom && y2 > m_Clipper.bottom)) return;
	if(x1 == x2) { DrawVLine(x1, y1, y2, color); return; }
	if(y1 == y2) { DrawHLine(x1, x2, y1, color); return; }

	int x_diff = x2-x1, y_diff = y2-y1;
	if((x1 < m_Clipper.left || x1 > m_Clipper.right) || (y1 < m_Clipper.top || y1 > m_Clipper.bottom))
	{
		if(x1 < m_Clipper.left) { y1 += ((y_diff)*(m_Clipper.left-x1))/x_diff; x1 = m_Clipper.left; }
		if(y1 < m_Clipper.top) { x1 += ((x_diff)*(m_Clipper.top-y1))/y_diff; y1 = m_Clipper.top; }
		if(x1 > m_Clipper.right)	{ y1 -= ((y_diff)*(x1-m_Clipper.right))/x_diff; x1 = m_Clipper.right;	}
		if(y1 > m_Clipper.bottom) { x1 -= ((x_diff)*(y1-m_Clipper.bottom))/y_diff; y1 = m_Clipper.bottom; }
	}
	if((x2 < m_Clipper.left || x2 > m_Clipper.right) || (y2 < m_Clipper.top || y2 > m_Clipper.bottom))
	{
		if(x2 < m_Clipper.left) { y2 += ((y_diff)*(m_Clipper.left-x2))/x_diff; x2 = m_Clipper.left; }
		if(y2 < m_Clipper.top) { x2 += ((x_diff)*(m_Clipper.top-y2))/y_diff; y2 = m_Clipper.top; }
		if(x2 > m_Clipper.right) { y2 -= ((y_diff)*(x2-m_Clipper.right))/x_diff; x2 = m_Clipper.right; }
		if(y2 > m_Clipper.bottom) { x2 -= ((x_diff)*(y2-m_Clipper.bottom))/y_diff; y2 = m_Clipper.bottom; }
	}

	int x_unit, y_unit;
	int error_term = 0;
	char *address = m_Address+x1+y1*m_Pitch;
	x_diff = x2-x1;
	y_diff = y2-y1;
	
	if(y_diff < 0) { y_diff=-y_diff; y_unit = -m_Pitch; }
	else { y_unit = m_Pitch; }
	if(x_diff < 0) { x_diff=-x_diff; x_unit = -1; }
	else { x_unit = 1; }

	if(x_diff == y_diff)
	{
		int length = x_diff+1;
		for(int i = 0; i < length; i++)
		{
			*address = color;
			address += x_unit;
			address += y_unit;
		}
	}
	else if(x_diff > y_diff)
	{
		int length = x_diff+1;
		for(int i = 0; i < length; i++)
		{
			*address = color;
			address += x_unit;
			error_term += y_diff;
			if(error_term > x_diff)
			{
				error_term -= x_diff;
				address += y_unit;
			}
		}
	}
	else
	{
		int length = y_diff+1;
		for(int i = 0; i < length; i++)
		{
			*address = color;
			address += y_unit;
			error_term += x_diff;
			if(error_term > y_diff) 
			{ 
				error_term -= y_diff;
				address += x_unit;
			}
		}
	}
}

void XBuffer::DrawBlankRectangle(XRectangle rectangle, char color)
{
	DrawVLine(rectangle.left, rectangle.top, rectangle.bottom, color);
	DrawVLine(rectangle.right, rectangle.top, rectangle.bottom, color);
	DrawHLine(rectangle.left, rectangle.right, rectangle.top, color);
	DrawHLine(rectangle.left, rectangle.right, rectangle.bottom, color);
}

void XBuffer::DrawBlankRectangle(int left, int top, int right, int bottom, char color)
{
	DrawVLine(left, top, bottom, color);
	DrawVLine(right, top, bottom, color);
	DrawHLine(left, right, top, color);
	DrawHLine(left, right, bottom, color);
}

void XBuffer::DrawFillRectangle(XRectangle rectangle, char color)
{
	DrawFillRectangle(rectangle.left, rectangle.top, rectangle.right, rectangle.bottom, color);
}

void XBuffer::DrawFillRectangle(int left, int top, int right, int bottom, char color)
{
	XRectangle rectangle(left, top, right, bottom);
	if(rectangle.CheckCrash(m_Clipper) != TRUE) return;
	rectangle.ClipEdge(m_Clipper);

	int width_count = rectangle.GetWidth();
	int height_count = rectangle.GetHeight();
	int add_width = m_Pitch-width_count;
	char *address = m_Address+left+top*m_Pitch;

	__asm
	{
		mov edi, address
		mov al, color
		mov ah, color
		push ax
		push ax
		pop eax
		mov edx, height_count
goto1:	mov ecx, width_count
		shr ecx, 1
		jnc goto2
		stosb
goto2:	shr ecx, 1
		jnc goto3
		stosw
goto3:	rep stosd
		add edi, add_width
		dec edx
		jnz goto1
	}
}

void XBuffer::DrawEllipse(int middle_x, int middle_y, int length_x, int length_y, char color)
{
	XRectangle border(middle_x-length_x,middle_y-length_y,middle_x+length_x,middle_y+length_y);
	if(border.CheckCrash(GetClipper()) != TRUE) return;
	int x = 0, y = length_y;
	long a = length_x, b = length_y;
	long a_square = a * a;
	long two_a_square = a_square << 1;
	long b_square = b * b;
	long two_b_square = b_square << 1;
	long d, dx, dy;
	d = b_square - a_square*b + (a_square>>2);
	dx = 0;
	dy = two_a_square * b;
	while( dx < dy )
	{
		DrawPixel(middle_x+x, middle_y+y, color);
	    DrawPixel(middle_x-x, middle_y+y, color);
		DrawPixel(middle_x+x, middle_y-y, color);
	    DrawPixel(middle_x-x, middle_y-y, color);
		if( d > 0 )
		{
			y--;
			dy -= two_a_square;
			d -= dy;
		}
		x++;
		dx += two_b_square;
		d += b_square + dx;
	}
	d += ( 3*(a_square - b_square)/2 - (dx+dy)/2 );
	while( y >= 0 )
	{
	    DrawPixel(middle_x+x, middle_y+y, color);
		DrawPixel(middle_x-x, middle_y+y, color);
	    DrawPixel(middle_x+x, middle_y-y, color);
		DrawPixel(middle_x-x, middle_y-y, color);
		if( d < 0 )
		{
			x++;
			dx += two_b_square;
			d += dx;
		}
		y--;
		dy -= two_a_square;
		d += a_square - dy;
	}
}

void XBuffer::DrawFillEllipse(int middle_x, int middle_y, int length_x, int length_y, char color)
{
	XRectangle border(middle_x-length_x,middle_y-length_y,middle_x+length_x,middle_y+length_y);
	if(border.CheckCrash(GetClipper()) != TRUE) return;
	int x = 0, y = length_y;
	long a = length_x, b = length_y;
	long a_square = a * a;
	long two_a_square = a_square << 1;
	long b_square = b * b;
	long two_b_square = b_square << 1;
	long d, dx, dy;
	d = b_square - a_square*b + (a_square>>2);
	dx = 0;
	dy = two_a_square * b;
	while( dx < dy )
	{
		DrawHLine(middle_x+x, middle_x-x, middle_y+y, color);
		DrawHLine(middle_x+x, middle_x-x, middle_y-y, color);
		if( d > 0 )
		{
			y--;
			dy -= two_a_square;
			d -= dy;
		}
		x++;
		dx += two_b_square;
		d += b_square + dx;
	}
	d += ( 3*(a_square - b_square)/2 - (dx+dy)/2 );
	while( y >= 0 )
	{
		DrawHLine(middle_x+x, middle_x-x, middle_y+y, color);
		DrawHLine(middle_x+x, middle_x-x, middle_y-y, color);
		if( d < 0 )
		{
			x++;
			dx += two_b_square;
			d += dx;
		}
		y--;
		dy -= two_a_square;
		d += a_square - dy;
	}
}

void XBuffer::Blit(XBuffer *target, int x, int y, XRectangle rectangle)
{
	rectangle.ClipEdge(GetBorder());

	char *source_address = m_Address+rectangle.left+rectangle.top*m_Pitch;
	char *target_address = target->GetAddress();
	int blit_width = rectangle.GetWidth();
	int blit_height = rectangle.GetHeight();
	int source_add, target_add;
	int target_pitch = target->GetPitch();
	XRectangle target_clipper = target->GetClipper();
	if(x+blit_width-1 < target_clipper.left || x > target_clipper.right) return;
	if(y+blit_height-1 < target_clipper.top || y > target_clipper.bottom) return;
	if(x < target_clipper.left)
	{
		source_address += target_clipper.left-x;
		blit_width -= target_clipper.left-x;
		x = target_clipper.left;
	}
	if(x+blit_width-1 > target_clipper.right)
		blit_width -= (x+blit_width-1) - target_clipper.right;
	if(y < target_clipper.top)
	{
		source_address += (target_clipper.top-y)*m_Pitch;
		blit_height -= target_clipper.top-y;
		y = target_clipper.top;
	}
	if(y+blit_height-1 > target_clipper.bottom)
		blit_height -= (y+blit_height-1) - target_clipper.bottom;

	target_address += x + y * target_pitch;
	source_add = m_Pitch - blit_width;
	target_add = target_pitch - blit_width;

	__asm
	{
		mov esi, source_address
		mov edi, target_address
		mov edx, blit_height
goto1:	mov ecx, blit_width
		shr ecx, 1
		jnc goto2
		movsb
goto2:	shr ecx, 1
		jnc goto3
		movsw
goto3:	rep movsd
		add esi, source_add
		add edi, target_add
		dec edx
		jnz goto1
	}
}


