#include <windows.h>
#include "ddraw.h"
#include "etc_func.h"
#include "directdraw.h"

XOffScreen::XOffScreen()
{
	m_DirectDrawHandler = NULL;
	m_Surface = NULL;
}

XOffScreen::XOffScreen(XDirectDrawHandler *DDH, int width, int height, int blank_color_start, int blank_color_end)
{
	m_DirectDrawHandler = NULL;
	m_Surface = NULL;
	Create(DDH,width,height,blank_color_start,blank_color_end);
}

XOffScreen::~XOffScreen()
{
	if(m_Surface != NULL) m_Surface->Release();
}

int XOffScreen::Create(XDirectDrawHandler *DDH, int width, int height, int blank_color_start, int blank_color_end)
{
	if(DDH == NULL) return FALSE;
	if(m_Surface != NULL) return FALSE;

	if(width <= 0 || width > DDH->GetWidth()) return FALSE;
	if(height <= 0 || height > DDH->GetHeight()) return FALSE;

	m_DirectDrawHandler = DDH;

	m_Ddsd.dwSize = sizeof(m_Ddsd);
	m_Ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_CKSRCBLT;
	m_Ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	m_Ddsd.dwWidth = width;
	m_Ddsd.dwHeight = height;
	m_Ddsd.ddckCKSrcBlt.dwColorSpaceLowValue = blank_color_start;
	m_Ddsd.ddckCKSrcBlt.dwColorSpaceHighValue = blank_color_end;

	if(m_DirectDrawHandler->GetDirectDraw()->CreateSurface(&m_Ddsd, &m_Surface, NULL) != DD_OK) return FALSE;

	m_Width = width;
	m_Height = height;
	m_Buffer.SetWidth(m_Width);
	m_Buffer.SetHeight(m_Height);
	m_Buffer.SetPitch(m_Width);
	m_Buffer.SetClipper(m_Buffer.GetBorder());

	return TRUE;
}

int XOffScreen::ChangeSize(int width, int height)
{
	if(m_Surface == NULL) return FALSE;

	if(width <= 0 || width > m_DirectDrawHandler->GetWidth()) return FALSE;
	if(height <= 0 || height > m_DirectDrawHandler->GetHeight()) return FALSE;

	m_Surface->Release();
	m_Surface = NULL;

	m_Ddsd.dwWidth = width;
	m_Ddsd.dwHeight = height;

	if(m_DirectDrawHandler->GetDirectDraw()->CreateSurface(&m_Ddsd, &m_Surface, NULL) != DD_OK) return FALSE;

	m_Width = width;
	m_Height = height;
	m_Buffer.SetWidth(m_Width);
	m_Buffer.SetHeight(m_Height);
	m_Buffer.SetPitch(m_Width);
	m_Buffer.SetClipper(m_Buffer.GetBorder());

	return TRUE;
}

int XOffScreen::GetWidth()
{
	return m_Width;
}

int XOffScreen::GetHeight()
{
	return m_Height;
}

XRectangle XOffScreen::GetBorder()
{
	return XRectangle(0,0,m_Width-1,m_Height-1);
}

LPDIRECTDRAWSURFACE XOffScreen::GetSurface()
{
	return m_Surface;
}

XBuffer *XOffScreen::GetBuffer()
{
	MemorySet((char*)&m_Ddsd, 0, sizeof(m_Ddsd));
	m_Ddsd.dwSize = sizeof(m_Ddsd);
	while(m_Surface->Lock(NULL, &m_Ddsd, DDLOCK_SURFACEMEMORYPTR, NULL) != DD_OK);
	m_Buffer.SetAddress((char*)m_Ddsd.lpSurface);
	m_Buffer.SetPitch(m_Ddsd.lPitch);

	return &m_Buffer;
}

void XOffScreen::ReleaseBuffer()
{
	m_Surface->Unlock(NULL);
}

int XOffScreen::LoadBitmap(char *name)
{
	BITMAPFILEHEADER bfh;
	BITMAPINFOHEADER bih;

	if(LoadBitmapHeader(name, &bfh, &bih) == FALSE) return FALSE;
	if( m_Width != bih.biWidth || m_Height != bih.biHeight) 
		ChangeSize(bih.biWidth, bih.biHeight);
	
	XBuffer *temp = GetBuffer();
	int result = LoadBitmapDataForPitch(name, temp->GetAddress(), m_Ddsd.lPitch);
	ReleaseBuffer();
	return result;
}

XBigOffScreen::XBigOffScreen()
{
	m_Screen = NULL;
	m_DirectDrawHandler = NULL;
}

XBigOffScreen::XBigOffScreen(XDirectDrawHandler *DDH, int width, int height, int blank_color_start, int blank_color_end)
{
	m_Screen = NULL;
	m_DirectDrawHandler = NULL;
	Create(DDH, width, height, blank_color_start, blank_color_end);
}

XBigOffScreen::~XBigOffScreen()
{
	for(int i = 0; i < m_ScreenXCount * m_ScreenYCount; i++)
		delete m_Screen[i];
	delete [] m_Screen;
}

int XBigOffScreen::Create(XDirectDrawHandler *DDH, int width, int height, int blank_color_start, int blank_color_end)
{
	if(DDH == NULL) return FALSE;
	if(m_Screen != NULL) return FALSE;
	if(width <= 0 || height <= 0) return FALSE;

	int i, j;
	m_DirectDrawHandler = DDH;
	m_Width = width;
	m_Height = height;
	m_XEdgeFlag = 0;
	m_YEdgeFlag = 0;
	if(m_Width % GetSmallOffScreenWidth()) m_XEdgeFlag = 1;
	if(m_Height % GetSmallOffScreenHeight()) m_YEdgeFlag = 1;
	m_ScreenXCount = m_Width / GetSmallOffScreenWidth() + m_XEdgeFlag;
	m_ScreenYCount = m_Height / GetSmallOffScreenHeight() + m_YEdgeFlag;

	m_Screen = new XOffScreen*[m_ScreenXCount * m_ScreenYCount];
	for(j = 0; j < m_ScreenYCount-m_YEdgeFlag; j++)
		for(i = 0; i < m_ScreenXCount-m_XEdgeFlag; i++)
			m_Screen[i+j*m_ScreenXCount] = new XOffScreen(m_DirectDrawHandler,GetSmallOffScreenWidth(),GetSmallOffScreenHeight());
	if(m_XEdgeFlag)
	{
		i = m_ScreenXCount-m_XEdgeFlag;
		for(j = 0; j < m_ScreenYCount-m_YEdgeFlag; j++)
			m_Screen[i+j*m_ScreenXCount] = new XOffScreen(m_DirectDrawHandler,GetEdgeWidth(),GetSmallOffScreenHeight());
	}
	if(m_YEdgeFlag)
	{
		j = m_ScreenYCount-m_YEdgeFlag;
		for(i = 0; i < m_ScreenXCount-m_XEdgeFlag; i++)
			m_Screen[i+j*m_ScreenXCount] = new XOffScreen(m_DirectDrawHandler,GetSmallOffScreenWidth(),GetEdgeHeight());
	}
	if(m_XEdgeFlag && m_YEdgeFlag)
	{
		i = m_ScreenXCount-m_XEdgeFlag;
		j = m_ScreenYCount-m_YEdgeFlag;
		m_Screen[i+j*m_ScreenXCount] = new XOffScreen(m_DirectDrawHandler,GetEdgeWidth(),GetEdgeHeight());
	}

	return TRUE;
}

int XBigOffScreen::ChangeSize(int width, int height)
{
	for(int i = 0; i < m_ScreenXCount * m_ScreenYCount; i++)
		delete m_Screen[i];
	delete [] m_Screen;
	m_Screen = NULL;

	Create(m_DirectDrawHandler,width,height);

	return TRUE;
}

int XBigOffScreen::GetWidth()
{
	return m_Width;
}

int XBigOffScreen::GetHeight()
{
	return m_Height;
}

int XBigOffScreen::GetSmallOffScreenXCount()
{
	return m_ScreenXCount;
}

int XBigOffScreen::GetSmallOffScreenYCount()
{
	return m_ScreenYCount;
}

int XBigOffScreen::GetSmallOffScreenWidth()
{
	return m_DirectDrawHandler->GetWidth();
}

int XBigOffScreen::GetSmallOffScreenHeight()
{
	return m_DirectDrawHandler->GetHeight();
}

int XBigOffScreen::GetEdgeWidth()
{
	return m_Width % m_DirectDrawHandler->GetWidth();
}

int XBigOffScreen::GetEdgeHeight()
{
	return m_Height % m_DirectDrawHandler->GetHeight();
}

XRectangle XBigOffScreen::GetBorder()
{
	return XRectangle(0, 0, m_Width-1, m_Height-1);
}

XOffScreen *XBigOffScreen::GetSmallOffScreen(int x, int y)
{
	if(x < 0 || x >= m_ScreenXCount || y < 0 || y >= m_ScreenYCount) return NULL;
	return m_Screen[x+y*m_ScreenXCount];
}

int XBigOffScreen::LoadBitmap(char *name)
{
	XBuffer source(1,1), *target;
	if(source.LoadBitmap(name) == FALSE) return FALSE;
	if(source.GetWidth() != m_Width || source.GetHeight() != m_Height)
	{
		if(ChangeSize(source.GetWidth(),source.GetHeight()) == FALSE) return FALSE;
	}

	XRectangle a;
	int i, j;
	for(j = 0; j < m_ScreenYCount-m_YEdgeFlag; j++)
	{
		for(i = 0; i < m_ScreenXCount-m_XEdgeFlag; i++)
		{
			target = GetSmallOffScreen(i,j)->GetBuffer();
			source.Blit(target,0,0,XRectangle(i*GetSmallOffScreenWidth(),j*GetSmallOffScreenHeight(),(i+1)*GetSmallOffScreenWidth()-1,(j+1)*GetSmallOffScreenHeight()-1));
			GetSmallOffScreen(i,j)->ReleaseBuffer();
		}
	}
	if(m_XEdgeFlag)
	{
		i = m_ScreenXCount-m_XEdgeFlag;
		for(j = 0; j < m_ScreenYCount-m_YEdgeFlag; j++)
		{
			target = GetSmallOffScreen(i,j)->GetBuffer();
			source.Blit(target,0,0,XRectangle(i*GetSmallOffScreenWidth(),j*GetSmallOffScreenHeight(),i*GetSmallOffScreenWidth()+GetEdgeWidth()-1,(j+1)*GetSmallOffScreenHeight()-1));
			GetSmallOffScreen(i,j)->ReleaseBuffer();
		}
	}
	if(m_YEdgeFlag)
	{
		j = m_ScreenYCount-m_YEdgeFlag;
		for(i = 0; i < m_ScreenXCount-m_XEdgeFlag; i++)
		{
			target = GetSmallOffScreen(i,j)->GetBuffer();
			source.Blit(target,0,0,XRectangle(i*GetSmallOffScreenWidth(),j*GetSmallOffScreenHeight(),(i+1)*GetSmallOffScreenWidth()-1,j*GetSmallOffScreenHeight()+GetEdgeHeight()-1));
			GetSmallOffScreen(i,j)->ReleaseBuffer();
		}
	}
	if(m_XEdgeFlag && m_YEdgeFlag)
	{
		i = m_ScreenXCount-m_XEdgeFlag;
		j = m_ScreenYCount-m_YEdgeFlag;
		target = GetSmallOffScreen(i,j)->GetBuffer();
		source.Blit(target,0,0,XRectangle(i*GetSmallOffScreenWidth(),j*GetSmallOffScreenHeight(),i*GetSmallOffScreenWidth()+GetEdgeWidth()-1,j*GetSmallOffScreenWidth()+GetEdgeHeight()-1));
		GetSmallOffScreen(i,j)->ReleaseBuffer();
	}

	return TRUE;
}

void XBigOffScreen::Blit()
{
}

XDirectDrawHandler::XDirectDrawHandler()
{
	m_DirectDraw = NULL;
	m_PrimarySurface = NULL;
	m_BackSurface = NULL;
	m_Clipper = NULL;
}

XDirectDrawHandler::~XDirectDrawHandler()
{
	if(m_DirectDraw != NULL)
	{
		if(m_BackSurface != NULL) m_BackSurface->Release();
		if(m_PrimarySurface != NULL) m_PrimarySurface->Release();
		if(m_Clipper != NULL) m_Clipper->Release();
		m_DirectDraw->Release();
	}
}

int XDirectDrawHandler::Create(HWND hWnd, bool full_screen, DWORD dwDisplayWidth, DWORD dwDisplayHeight, DWORD dwBitPlanes, DWORD RefreshRate)
{
	DDSCAPS ddscaps;

	m_hWnd = hWnd;
	m_Width = dwDisplayWidth;
	m_Height = dwDisplayHeight;
	m_FullScreenFlag = full_screen;

	if(DirectDrawCreate(NULL,&m_DirectDraw,NULL) != DD_OK) return FALSE;
	if(m_DirectDraw->SetCooperativeLevel(m_hWnd, (m_FullScreenFlag == TRUE) ? (DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWREBOOT) : (DDSCL_NORMAL)) != DD_OK) return FALSE;

	if(m_FullScreenFlag == TRUE)
		if(m_DirectDraw->SetDisplayMode(m_Width, m_Height, dwBitPlanes) != DD_OK) return FALSE;

	m_Ddsd.dwSize = sizeof(m_Ddsd);
	m_Ddsd.dwFlags = DDSD_CAPS;
	m_Ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
	m_Ddsd.dwBackBufferCount = 1;
	if(m_FullScreenFlag == TRUE)
	{
		m_Ddsd.dwFlags |= DDSD_BACKBUFFERCOUNT;
		m_Ddsd.ddsCaps.dwCaps |= DDSCAPS_FLIP | DDSCAPS_COMPLEX;
	}
	if(m_DirectDraw->CreateSurface(&m_Ddsd, &m_PrimarySurface, NULL) != DD_OK) return FALSE;

	if(m_FullScreenFlag == TRUE)
	{
		ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
		if(m_PrimarySurface->GetAttachedSurface(&ddscaps,&m_BackSurface) != DD_OK) return FALSE;
	}
	else
	{
		m_Ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
		m_Ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
		m_Ddsd.dwWidth = m_Width;
		m_Ddsd.dwHeight = m_Height;
		if(m_DirectDraw->CreateSurface(&m_Ddsd, &m_BackSurface, NULL) != DD_OK) return FALSE;
	}

	if(m_FullScreenFlag == FALSE)
	{
		if(m_DirectDraw->CreateClipper(0, &m_Clipper, NULL) != DD_OK) return FALSE;
		if(m_Clipper->SetHWnd(0, m_hWnd) != DD_OK) return FALSE;
		if(m_PrimarySurface->SetClipper(m_Clipper) != DD_OK) return FALSE;
	}

	m_Buffer.SetWidth(m_Width);
	m_Buffer.SetHeight(m_Height);
	m_Buffer.SetClipper(m_Buffer.GetBorder());
	GetBackSurfaceBuffer()->Clear(0);
	ReleaseBackSurfaceBuffer();

	if(m_DirectDraw->CreatePalette(DDPCAPS_8BIT, m_PaletteEntry, &m_Palette, NULL) != DD_OK) return FALSE;

	return TRUE;
}

LPDIRECTDRAW XDirectDrawHandler::GetDirectDraw()
{
	return m_DirectDraw; 
}

int XDirectDrawHandler::GetWidth()
{
	return m_Width;
}

int XDirectDrawHandler::GetHeight()
{
	return m_Height;
}

XBuffer *XDirectDrawHandler::GetBackSurfaceBuffer()
{ 
	m_Ddsd.dwSize = sizeof(m_Ddsd);
	while(m_BackSurface->Lock(NULL, &m_Ddsd, DDLOCK_SURFACEMEMORYPTR, NULL) != DD_OK);
	m_Buffer.SetAddress((char*)m_Ddsd.lpSurface);
	m_Buffer.SetPitch(m_Ddsd.lPitch);
	return &m_Buffer;
}

void XDirectDrawHandler::ReleaseBackSurfaceBuffer()
{ 
	m_BackSurface->Unlock(NULL);
}

LPDIRECTDRAWSURFACE XDirectDrawHandler::GetPrimarySurface()
{
	return m_PrimarySurface;
}

LPDIRECTDRAWSURFACE XDirectDrawHandler::GetBackSurface()
{
	return m_BackSurface;
}

int XDirectDrawHandler::LoadBitmapPalette(char *name)
{
	return ::LoadBitmapPalette(name, m_PaletteEntry);
}

int XDirectDrawHandler::SetPalette()
{
	m_Palette->SetEntries(0,0,256,m_PaletteEntry);
	return m_PrimarySurface->SetPalette(m_Palette);
}
