unit Player;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, StdCtrls, Buttons, ExtCtrls, ComCtrls, XaudioPlayer;

type
  TPlayerForm = class(TForm)
    PlayButton: TBitBtn;
    MainMenu: TMainMenu;
    StopButton: TBitBtn;
    PauseButton: TBitBtn;
    MainMenu_File: TMenuItem;
    MainMenu_File_OpenFileItem: TMenuItem;
    MainMenu_File_OpenURLItem: TMenuItem;
    VolumeButton: TSpeedButton;
    OpenFileDialog: TOpenDialog;
    MainBevel: TBevel;
    MainMenu_Help: TMenuItem;
    MainMenu_Help_AboutItem: TMenuItem;
    InputNameText: TStaticText;
    TimecodeText: TStaticText;
    DurationText: TStaticText;
    StreamInfoText: TStaticText;
    StatusBar: TStatusBar;
    XaudioPlayer: TXaudioPlayer;
    MainMenu_Windows: TMenuItem;
    MainMenu_Windows_ShowInfo: TMenuItem;
    MainMenu_Windows_Equalizer: TMenuItem;
    TrackBar: TTrackBar;
    procedure MainMenu_File_OpenFileItemClick(Sender: TObject);
    procedure XaudioPlayerNotifyInputName(Sender: TXaudioPlayer;
      const Name: String);
    procedure XaudioPlayerNotifyInputState(Sender: TXaudioPlayer;
      State: Integer);
    procedure MainMenu_File_OpenURLItemClick(Sender: TObject);
    procedure XaudioPlayerNotifyNack(Sender: TXaudioPlayer; Command: Byte;
      Code: Smallint);
    procedure XaudioPlayerNotifyProgress(Sender: TXaudioPlayer; Source,
      Code: Byte; Value: Smallint; const Msg: String);
    procedure XaudioPlayerNotifyInputPosition(Sender: TXaudioPlayer;
      Offset, Range: Cardinal);
    procedure XaudioPlayerNotifyInputTimecode(Sender: TXaudioPlayer; Hours,
      Minutes, Seconds, Fractions: Byte);
    procedure FormDestroy(Sender: TObject);
    procedure VolumeButtonClick(Sender: TObject);
    procedure XaudioPlayerNotifyInputStreamInfo(Sender: TXaudioPlayer;
      Level, Layer: Byte; Bitrate, Frequency: Word; Mode: Byte);
    procedure PlayButtonClick(Sender: TObject);
    procedure PauseButtonClick(Sender: TObject);
    procedure StopButtonClick(Sender: TObject);
    procedure MainMenu_Windows_ShowInfoClick(Sender: TObject);
    procedure XaudioPlayerNotifyInputModule(Sender: TXaudioPlayer;
      ID: SmallInt);
    procedure XaudioPlayerNotifyOutputState(Sender: TXaudioPlayer;
      State: Integer);
    procedure XaudioPlayerNotifyOutputVolume(Sender: TXaudioPlayer;
      MasterLevel, PcmLevel, Balance: Byte);
    procedure XaudioPlayerNotifyOutputPcmLevel(Sender: TXaudioPlayer;
      PcmLevel: Byte);
    procedure XaudioPlayerNotifyOutputMasterLevel(Sender: TXaudioPlayer;
      MasterLevel: Byte);
    procedure XaudioPlayerNotifyOutputBalance(Sender: TXaudioPlayer;
      Balance: Byte);
    procedure XaudioPlayerNotifyOutputName(Sender: TXaudioPlayer;
      const Name: String);
    procedure XaudioPlayerNotifyOutputModule(Sender: TXaudioPlayer;
      ID: SmallInt);
    procedure MainMenu_Windows_EqualizerClick(Sender: TObject);
    procedure XaudioPlayerNotifyInputDuration(Sender: TXaudioPlayer;
      Duration: Cardinal);
    procedure XaudioPlayerNotifyPlayerState(Sender: TXaudioPlayer;
      State: Integer);
    procedure XaudioPlayerNotifyReady(Sender: TObject);
    procedure TrackBarChange(Sender: TObject);
    procedure XaudioPlayerNotifyInputModuleInfo(Sender: TXaudioPlayer; ID,
      NbDevices: Byte; const Name, Description: String);
    procedure XaudioPlayerNotifyInputDeviceInfo(Sender: TXaudioPlayer;
      ModuleID, Index, Flags: Byte; const Name, Description: String);
    procedure XaudioPlayerNotifyOutputDeviceInfo(Sender: TXaudioPlayer;
      ModuleID, Index, Flags: Byte; const Name, Description: String);
    procedure XaudioPlayerNotifyOutputModuleInfo(Sender: TXaudioPlayer; ID,
      NbDevices: Byte; const Name, Description: String);
  private
    { Private declarations }
    Tracking: Boolean;
  public
    { Public declarations }
  end;

var
  PlayerForm: TPlayerForm;

implementation

uses URLEntry, Volume, Info, Equalizer, Modules;

{$R *.DFM}

procedure ShowError(Text: String; Code: Integer);
var
  Msg: String;
begin
  Msg := Text + ' (' + PlayerForm.XaudioPlayer.ErrorString(code) + ')';
  MessageDlg(Msg, mtError, [mbOK], 0);
end;

procedure TPlayerForm.MainMenu_File_OpenFileItemClick(Sender: TObject);
begin
  if OpenFileDialog.Execute then
  begin
    XaudioPlayer.InputOpen(OpenFileDialog.FileName);
  end
end;

procedure TPlayerForm.XaudioPlayerNotifyInputName(Sender: TXaudioPlayer;
  const Name: String);
begin
  InputNameText.Caption := Name;
  InfoForm.InputNameText.Caption := Name;
end;

procedure TPlayerForm.XaudioPlayerNotifyInputState(Sender: TXaudioPlayer;
  State: Integer);
begin
  if State = XA_INPUT_STATE_OPEN then
    begin
      PlayButton.Enabled := True;
      PauseButton.Enabled := True;
      StopButton.Enabled := True;
      XaudioPlayer.Play;
    end
  else
    begin
      PlayButton.Enabled := False;
      PauseButton.Enabled := False;
      StopButton.Enabled := False;
    end;
end;

procedure TPlayerForm.MainMenu_File_OpenURLItemClick(Sender: TObject);
begin
  URLEntryForm.ShowModal;
end;

procedure TPlayerForm.XaudioPlayerNotifyNack(Sender: TXaudioPlayer;
  Command: Byte; Code: Smallint);
begin
  case Command of
  XA_MSG_COMMAND_INPUT_OPEN:
    ShowError('Cannot Open Input', Code);

  XA_MSG_COMMAND_PLAY:
    ShowError('Cannot Play Stream', Code);
  end;

end;

procedure TPlayerForm.XaudioPlayerNotifyProgress(Sender: TXaudioPlayer;
  Source, Code: Byte; Value: Smallint; const Msg: String);
begin
  case Code of
    16: StatusBar.Panels[0].Text := Msg;
    17: StatusBar.Panels[0].Text := Msg;
  end;
end;

procedure TPlayerForm.XaudioPlayerNotifyInputPosition(
  Sender: TXaudioPlayer; Offset, Range: Cardinal);
begin
  TrackBar.Position := Offset;
end;

procedure TPlayerForm.XaudioPlayerNotifyInputTimecode(
  Sender: TXaudioPlayer; Hours, Minutes, Seconds, Fractions: Byte);
begin
  TimecodeText.Caption := Format('%.2d:%.2d:%.2d', [Hours, Minutes, Seconds]);
end;

procedure TPlayerForm.FormDestroy(Sender: TObject);
begin
  XaudioPlayer.OutputReset;
  XaudioPlayer.Exit;
end;

procedure TPlayerForm.VolumeButtonClick(Sender: TObject);
begin
  VolumeForm.Show;
end;

procedure TPlayerForm.XaudioPlayerNotifyInputStreamInfo(
  Sender: TXaudioPlayer; Level, Layer: Byte; Bitrate, Frequency: Word;
  Mode: Byte);
begin
  StreamInfoText.Caption := 'MPEG ' + IntToStr(Level) + ', Layer ' + IntToStr(Layer) + ', ' + IntToStr(Bitrate) + ' Kbps, ' + IntToStr(Frequency) + ' Hz';
end;

procedure TPlayerForm.PlayButtonClick(Sender: TObject);
begin
  XaudioPlayer.Play;
end;

procedure TPlayerForm.PauseButtonClick(Sender: TObject);
begin
  XaudioPlayer.Pause
end;

procedure TPlayerForm.StopButtonClick(Sender: TObject);
begin
  XaudioPlayer.Stop;
end;

procedure TPlayerForm.MainMenu_Windows_ShowInfoClick(Sender: TObject);
begin
  InfoForm.Show;
end;

procedure TPlayerForm.XaudioPlayerNotifyInputModule(Sender: TXaudioPlayer;
  ID: SmallInt);
begin
  InfoForm.InputModuleText.Caption := IntToStr(ID);
end;

procedure TPlayerForm.XaudioPlayerNotifyOutputState(Sender: TXaudioPlayer;
  State: Integer);
begin
  if (state = XA_OUTPUT_STATE_OPEN) then
    VolumeForm.Enabled := True
  else
    VolumeForm.Enabled := False;
end;

procedure TPlayerForm.XaudioPlayerNotifyOutputVolume(Sender: TXaudioPlayer;
  MasterLevel, PcmLevel, Balance: Byte);
begin
  VolumeForm.MasterTrackBar.Position := MasterLevel;
  VolumeForm.PcmTrackBar.Position := PcmLevel;
  VolumeForm.BalanceTrackBar.Position := Balance;
end;

procedure TPlayerForm.XaudioPlayerNotifyOutputPcmLevel(
  Sender: TXaudioPlayer; PcmLevel: Byte);
begin
  VolumeForm.PcmTrackBar.Position := PcmLevel;
end;

procedure TPlayerForm.XaudioPlayerNotifyOutputMasterLevel(
  Sender: TXaudioPlayer; MasterLevel: Byte);
begin
  VolumeForm.MasterTrackBar.Position := MasterLevel;
end;

procedure TPlayerForm.XaudioPlayerNotifyOutputBalance(
  Sender: TXaudioPlayer; Balance: Byte);
begin
  VolumeForm.BalanceTrackBar.Position := Balance;
end;

procedure TPlayerForm.XaudioPlayerNotifyOutputName(Sender: TXaudioPlayer;
  const Name: String);
begin
  InfoForm.InputNameText.Caption := Name;
end;

procedure TPlayerForm.XaudioPlayerNotifyOutputModule(Sender: TXaudioPlayer;
  ID: SmallInt);
begin
    InfoForm.OutputModuleText.Caption := IntToStr(ID);
end;

procedure TPlayerForm.MainMenu_Windows_EqualizerClick(Sender: TObject);
begin
    EqualizerForm.Show;
end;

procedure TPlayerForm.XaudioPlayerNotifyInputDuration(
  Sender: TXaudioPlayer; Duration: Cardinal);
var
  Hours: Integer;
  Minutes: Integer;
  Seconds: Integer;
begin
  Hours := Duration div 3600;
  Duration := Duration - 3600*Hours;
  Minutes := Duration div 60;
  Duration := Duration - 60*Minutes;
  Seconds := Duration;
  DurationText.Caption := Format('%.2d:%.2d:%.2d', [Hours, Minutes, Seconds]);
end;

procedure TPlayerForm.XaudioPlayerNotifyPlayerState(Sender: TXaudioPlayer;
  State: Integer);
begin
  case State of
    XA_PLAYER_STATE_STOPPED:
      begin
        StatusBar.Panels[1].Text := 'STOPPED';
        XaudioPlayer.InputSeek(0, 1);
      end;
    XA_PLAYER_STATE_PLAYING:
      StatusBar.Panels[1].Text := 'PLAYING';
    XA_PLAYER_STATE_PAUSED:
      StatusBar.Panels[1].Text := 'PAUSED';
    XA_PLAYER_STATE_EOF:
      begin
        { if we had a playlist, now would be the time to tell }
        { the player to open the next item in the list by     }
        { calling InputOpen() with the next name. This would  }
        { give a perfect queuing effect (no gap between songs)}
        { Since we do not have a playlist, we'll just rewind  }
        { the stream to the beginning.                        }
        StatusBar.Panels[1].Text := 'EOF';
        XaudioPlayer.InputSeek(0,1);
      end;
  end;
end;

procedure TPlayerForm.XaudioPlayerNotifyReady(Sender: TObject);
begin
  Tracking := False;
end;

procedure TPlayerForm.TrackBarChange(Sender: TObject);
begin
  XaudioPlayer.InputSeek(TrackBar.Position, TrackBar.Max);
end;

procedure TPlayerForm.XaudioPlayerNotifyInputModuleInfo(
  Sender: TXaudioPlayer; ID, NbDevices: Byte; const Name,
  Description: String);
begin
  ModulesForm.Show;
  ModulesForm.TreeView.Items.AddChild(ModulesForm.TreeView.Items[0],
    String(Name) + ' [' + String(Description) + ']');
  ModulesForm.TreeView.Items.Item[0].Expand(True);
end;

procedure TPlayerForm.XaudioPlayerNotifyInputDeviceInfo(
  Sender: TXaudioPlayer; ModuleID, Index, Flags: Byte; const Name,
  Description: String);
begin
  ModulesForm.TreeView.Items.AddChild(ModulesForm.TreeView.Items[0].Item[ModuleID],
    String(Name) + ' [' + String(Description) + ']');
  ModulesForm.TreeView.Items.Item[0].Expand(True);
end;

procedure TPlayerForm.XaudioPlayerNotifyOutputModuleInfo(
  Sender: TXaudioPlayer; ID, NbDevices: Byte; const Name,
  Description: String);
begin
  ModulesForm.Show;
  ModulesForm.TreeView.Items.AddChild(ModulesForm.TreeView.Items[0],
    String(Name) + ' [' + String(Description) + ']');
  ModulesForm.TreeView.Items.Item[0].Expand(True);
end;

procedure TPlayerForm.XaudioPlayerNotifyOutputDeviceInfo(
  Sender: TXaudioPlayer; ModuleID, Index, Flags: Byte; const Name,
  Description: String);
begin
  ModulesForm.TreeView.Items.AddChild(ModulesForm.TreeView.Items[0].Item[ModuleID],
    String(Name) + ' [' + String(Description) + ']');
  ModulesForm.TreeView.Items.Item[0].Expand(True);
end;

end.
