#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include "cAssembler.h"

///////////////////////// Constructer /////////////////////////
cAssembler::cAssembler(){
}

cAssembler::~cAssembler(){
	Close();
}
/////////////////////////// Close //////////////////////////////
void cAssembler::Close(){
}
/////////////////////////// Run ////////////////////////////////
int cAssembler::Run(int Params, char **Param){
	Title();
	if(!ProcessParameter(Params, Param)) return 1;

	if(!OperatorTable.Load("opcode.tbl")){
		printf("Operator Table Load Failure.\n");
		return 1;
	}

	if(!FirstPass()){
		printf("Error on first pass.\n");
		return 1;
	}
	UserSymbolTable.Print();

	if(!SecondPass()){
		printf("Error on second pass.\n");
		return 1;
	}
	puts("The Program Translated Completely.");
	return 0;

}
////////////////////////// Title ////////////////////////////////
void cAssembler::Title(){
	printf(" The System Programming Term Project\n");
	printf("[ Baby Assembler For 8086 Processor ]\n");
	printf("                       1997.5.30 Sat\n");
	printf("   Department of Computer Engineering\n");
	printf("          No. 96034001 Koo, Yeon-Hoon\n\n");
}

////////////////////////// Process Parameter //////////////////////////// 
bool cAssembler::ProcessParameter(int Params, char **Param){
	char *SourceFileName;
	char *ObjectFileName;
	if(Params < 2){
		printf("USAGE:\n");
		printf("BASM Source[.ASM] [Object[.COM]]\n");
		printf("Source : 8086 Assembly Program\n");
		printf("Object : Executable COM File\n");
		return false; 
	}

	SourceFileName = new char[strlen(Param[1]) + 5];
	strcpy(SourceFileName, Param[1]);
	if(strchr(SourceFileName, '.') == NULL){
		strcat(SourceFileName, ".asm");
	}
	if(Params > 2){
		ObjectFileName = new char[strlen(Param[2]) + 5];
		strcpy(ObjectFileName, Param[2]);
		if(strchr(ObjectFileName, '.') == NULL){
			strcat(ObjectFileName, ".com");
		}
	}else{
		ObjectFileName = new char[strlen(SourceFileName) + 1];
		strcpy(ObjectFileName, SourceFileName);
		strcpy(strchr(ObjectFileName, '.'), ".com");
	}

	SourceFile.SetFileName(SourceFileName);
	ObjectFile.SetFileName(ObjectFileName);

	printf("Source File: %s\n", SourceFileName);
	printf("Object File: %s\n", ObjectFileName);

	delete [] SourceFileName;
	delete [] ObjectFileName;

	return true;

}

/////////////////////////// First Pass ////////////////////////////
bool cAssembler::FirstPass(){
	char Line[256], Label[256], Operator[256], Operand[256];
	eDirective Directive;
	int ErrorCount = 0, LineNumber = 0, Value;

	// Initialize Variables
	Address = 0;
	IsFirstPass = true;

	if(!SourceFile.Open("rt")) return false;

	printf("\t\t\t\t[ First Pass ]\n");
	printf("===============================================================================\n");
	printf("LINE ADDR                          PROGRAM\n"); 
	printf("-------------------------------------------------------------------------------\n");
	while(true){
		LineNumber ++;
		if(SourceFile.Gets(Line, 256) == NULL){
			printf("%04d %04X\n", LineNumber, Address);
			break;
		}
		printf("%04d %04X %s", LineNumber, Address, Line);
		String = Line;
		LineMode = ExtractLine(Label, Operator, Directive, Operand);
		if(LineMode == LM_ERROR){
			printf("Error: Syntax error.\n");
			ErrorCount ++;
			continue;
		}

		if(LineMode == LM_NONE) continue;
		if(LineMode == LM_ERROR){
			ErrorCount ++;
		}else{
			if(LineMode & LM_LABEL)
				if(!((LineMode & LM_DIRECTIVE) && Directive == D_EQU))
					UserSymbolTable.Add(Label, Address);
			if(LineMode & LM_DIRECTIVE){
				if(!ProcessFirstDirective(Label, Directive, Operand))
					ErrorCount ++;
			}
			if(LineMode & LM_OPERATOR){
				if(GetMachineCodeSize(Operator, Operand, Value)){
					Address += Value;
				}else{
					ErrorCount ++;
				}
			}
		}
	}
	SourceFile.Close();
	IsFirstPass = false;
	printf("-------------------------------------------------------------------------------\n");
	printf("End of First Pass\tTotal %d Errors.\n", ErrorCount);
	printf("===============================================================================\n");
	if(ErrorCount > 0) return false;
	return true;
}
/////////////////////////// Second Pass //////////////////////
bool cAssembler::SecondPass(){
	char Line[256], Label[256], Operator[256], Operand[256];
	eDirective Directive;
	int LineNumber = 0, ErrorCount = 0;

	Address = 0;

	if(!SourceFile.Open("rt")) return false;
	if(!ObjectFile.Open("wb")){
		SourceFile.Close();
		return false;
	}

	printf("\t\t\t\t[ Second Pass ]\n");
	printf("===============================================================================\n");
	printf("LINE ADDR MACHINE CODE               PROGRAM\n"); 
	printf("-------------------------------------------------------------------------------\n");
	while(true){
		LineNumber ++;
		if(SourceFile.Gets(Line, 256) == NULL){
			printf("%04d %04X\n", LineNumber, Address);
			break;
		}
		String = Line;
		LineMode = ExtractLine(Label, Operator, Directive, Operand);
		printf("%04d %04X ", LineNumber, Address);

		if(LineMode == LM_ERROR){
			ErrorCount ++;
		}else{
			if(LineMode & LM_DIRECTIVE){
				if(!ProcessSecondDirective(Label, Directive, Operand))
					ErrorCount ++;
			}
			if(LineMode & LM_OPERATOR){
				if(!WriteMachineCode(Operator, Operand))
					ErrorCount ++;
			}else{
				printf("\t\t");
			}
		}
		printf("%s", Line);
	}
	printf("-------------------------------------------------------------------------------\n");
	printf("End of Second Pass\tTotal %d Errors.\n", ErrorCount);
	printf("===============================================================================\n");

	SourceFile.Close();
	ObjectFile.Close();
	return true;
}
///////////////////// Process First Directive //////////////////
bool cAssembler::ProcessFirstDirective(char *Label, eDirective Directive, char *Operand){
	int Value;

	switch(Directive){
	case D_DB:	// Define Byte
		if(!GetImmCount(Operand, Value)) return false;
		Address += Value;
		break;
	case D_DBS:	// Define Byte Storage
		if(!Solve(Operand, Value)) return false;
		Address += Value;
		break;
	case D_DW:	// Define Word
		if(!GetImmCount(Operand, Value)) return false;
		Address += Value * 2;
		break;
	case D_DWS:	// Define Word Storage
		if(!Solve(Operand, Value)) return false;
		Address += Value * 2;
		break;
	case D_DS: // Define String
		if(!GetStringSize(Operand, Value)) return false;
		Address += Value;
		break;
	case D_EQU:
		if(!(LineMode & LM_LABEL)) return false;
		if(!Solve(Operand, Value)) return false;
		UserSymbolTable.Add(Label, Value);
		break;
	case D_ORG:
		if(!Solve(Operand, Value)) return false;
		Address = Value;
		break;
	}
	return true;
}

bool cAssembler::ProcessSecondDirective(char *Label, eDirective Directive, char *Operand){
	int Value;

	switch(Directive){
	case D_DB:	// Define Byte
		if(!WriteByte(Operand, Value)) return false;
		Address += Value;
		break;
	case D_DBS:	// Define Byte Storage
		if(!Solve(Operand, Value)) return false;
		Address += Value;
		if(ObjectFile.WriteSpace(Value) == EOF) return false;
		break;
	case D_DW:	// Define Word
		if(!WriteWord(Operand, Value)) return false;
		Address += Value;
		break;
	case D_DWS:	// Define Word Storage
		if(!Solve(Operand, Value)) return false;
		Address += Value * 2;
		if(ObjectFile.WriteSpace(Value * 2) == EOF) return false;
		break;
	case D_DS: // Define String
		if(!WriteString(Operand, Value)) return false;
		Address += Value;
		break;
	case D_ORG:
		if(!Solve(Operand, Value)) return false;
		Address = Value;
		break;
	}
	return true;
}
/////////////////////// Get Machine Code Size //////////////////
bool cAssembler::GetMachineCodeSize(char *Operator, char *Operand, int &nSize){
	sOpCodeInfo OpCodeInfo;
	eOperandType Type1, Type2;

	String = Operand;

	OpCodeInfo.reg = 0;
	OpCodeInfo.sreg = 0;
	OpCodeInfo.imm = 0;
	OpCodeInfo.disp = 0;
	OpCodeInfo.mod = 0;
	OpCodeInfo.rm = 0;
	OpCodeInfo.w = false;
	OpCodeInfo.s = false;
	
	if(!SolveOperands(Type1, Type2, OpCodeInfo)) return false;

	if(OperatorTable.GetMachineCodeSize(Operator, Type1, Type2, OpCodeInfo, nSize)){
		return true;
	}
	return false;
}
/////////////////////////////// Write Machine Code /////////////
bool cAssembler::WriteMachineCode(char *Operator, char *Operand){
	byte Code[20];
	sOpCodeInfo OpCodeInfo;
	eOperandType Type1, Type2;
	int Size, i;

	String = Operand;

	OpCodeInfo.reg = 0;
	OpCodeInfo.sreg = 0;
	OpCodeInfo.imm = 0;
	OpCodeInfo.disp = 0;
	OpCodeInfo.mod = 0;
	OpCodeInfo.rm = 0;
	OpCodeInfo.w = false;
	OpCodeInfo.s = false;
	
	if(!SolveOperands(Type1, Type2, OpCodeInfo)) return false;

	if(OperatorTable.GetMachineCode(Operator, Type1, Type2, OpCodeInfo, Size, Address, Code)){
		ObjectFile.Write(Code, Size);
		PrintHexa(Code, Size);
		for(i = 0; i < (13 - Size); i ++){
			printf(" ");
		}
		Address += Size;
		return true;
	}
	Address += Size;
	return false;
}

/////////////////////////// String Methods ////////////////////
bool cAssembler::GetImmCount(char *nString, int &Count){
	int Value;

	String = nString;

	if(!SkipSpace()) return false;

	Count = 0;

	while(true){
		if(!SolveExpr(Value)) return false;
		Count ++;
		if(!SkipSpace()){
			return true;
		}
		if(*String != ',') return false;
		String ++;
	}
}
///////////////////////// Output File /////////////////////////
bool cAssembler::WriteByte(char *nString, int &Count){
	int Value;

	String = nString;
	Count = 0;

	if(!SkipSpace()) return false;

	while(true){
		if(!SolveExpr(Value)) return false;
		Count ++;
		if(ObjectFile.Putc(Value) == EOF) return false;
		printf("%01X", (unsigned)Value);
		if(!SkipSpace()){
			return true;
		}
		if(*String != ',') return false;
		String ++;
	}
}
bool cAssembler::WriteWord(char *nString, int &Count){
	int Value;
	int Low, High;

	String = nString;
	Count = 0;
	if(!SkipSpace()) return false;

	while(true){
		if(!SolveExpr(Value)) return false;
		Low = Value & 255;
		High = Value >> 8;
		if(ObjectFile.Putc(High) == EOF) return false;
		if(ObjectFile.Putc(Low) == EOF) return false;
		Count += 2;
		printf("%02X", (unsigned)Value);

		if(!SkipSpace()){
			return true;
		}
		if(*String != ',') return false;
		String ++;
	}
}

bool cAssembler::WriteString(char *nString, int &Count){
	char Temp[256];

	Count = ExtractString(Temp, nString);
	printf(Temp);
	return ObjectFile.Write(Temp, strlen(Temp)) != EOF;
}

///////////////////////// Parser /////////////////////////////
bool cAssembler::SolveOperands(eOperandType &Type1, eOperandType &Type2, sOpCodeInfo &OpCodeInfo){
	strlwr(String);

	Type1 = Type2 = OP_VOID;

	if(!SkipSpace()) return true;

	Type1 = SolveOperand(OpCodeInfo);
	if(Type1 == OP_ERROR){
		printf("Error: Operand syntax error.\n");
		return false;
	}

	if(!SkipSpace()) return true;

	if(*String == ','){
		String ++;
		if(Type1 == OP_REG || Type1 == OP_REG8 || Type1 == OP_REG16){
			OpCodeInfo.rm = OpCodeInfo.reg;
		}
		Type2 = SolveOperand(OpCodeInfo);
		if(Type2 == OP_ERROR){
			printf("Error: Right Operand need.\n");
			return false;
		}
	}
	return true;
}

eOperandType cAssembler::SolveOperand(sOpCodeInfo &OpCodeInfo){
	eOperandType Type;
	char *Temp;
	char Token1[100], Token2[100], Token3;
	bool IsShort = false, IsNear = false, IsWord = false;

	if(!SkipSpace()) return OP_ERROR;

	// Detect Label Mode
	if(String[0] == 's' && String[1] == 'h' && String[2] == 'o' && String[3] == 'r' && String[4] == 't'){
		IsShort = true;
		String += 5;
	}else if(String[0] == 'n' && String[1] == 'e' && String[2] == 'a' && String[3] == 'r'){
		IsNear = true;
		String += 4;
	}

	if(!SkipSpace()) return OP_ERROR;

	// Is Memory Reference?
	if(*String == '['){
		return SolveMemref(OpCodeInfo);
	} 
	Temp = String;
	// Detect Memory Mode
	if(sscanf(String, "%s %s %c", Token1, Token2, &Token3) == 3){
		if(Token3 == '[' && strcmp(Token2, "ptr") == 0){
			if(strcmp(Token1, "word") == 0){
				IsWord = true;
			}else if(strcmp(Token1, "byte") == 0){
				IsWord = false;
			}else{
				printf("Error: Wrong Prefix.\n");
				return OP_ERROR;
			}
			while(*String != '[') String ++;
			if(SolveMemref(OpCodeInfo) != OP_ERROR){
				if(IsWord) return OP_MEM16;
				else return OP_MEM8;
			}else return OP_ERROR;
		}
	}

	String = Temp;

	Type = SolveRegister(OpCodeInfo);
	if(Type != OP_ERROR) return Type;

	Type = SolveImm(OpCodeInfo);
	if(Type == OP_ERROR){
		printf("Error: Expression Error\n");
		return OP_ERROR;
	}
	if(IsNear) return OP_IMM16;
	return OP_IMM8;
}
eOperandType cAssembler::SolveRegister(sOpCodeInfo &OpCodeInfo){
	const int RegCount = 16;
	struct sRegister{
		char *Name;
		int reg;
		bool w;
	} RegList[] = {
		"al", 0, false, "cl", 1, false, "dl", 2, false, "bl", 3, false,
		"ah", 4, false, "ch", 5, false, "dh", 6, false, "bh", 7, false,
		"ax", 0, true, "cx", 1, true, "dx", 2, true, "bx", 3, true,
		"sp", 4, true, "bp", 5, true, "si", 6, true, "di", 7, true
	};
	int i;

	if(!SkipSpace()) return OP_ERROR;

	for(i = 0; i < RegCount; i ++){
		if(*String == RegList[i].Name[0] && *(String + 1) == RegList[i].Name[1]){
			OpCodeInfo.reg = RegList[i].reg;
			OpCodeInfo.w = RegList[i].w;
			String += 2;
			if(OpCodeInfo.w) return OP_REG16;
			return OP_REG8;
		}
	}
	if(*String == 'e' && *(String + 1) == 's') OpCodeInfo.sreg = 0;
	else if(*String == 'c' && *(String + 1) == 's') OpCodeInfo.sreg = 1;
	else if(*String == 's' && *(String + 1) == 's')	OpCodeInfo.sreg = 2;
	else if(*String == 'd' && *(String + 1) == 's')	OpCodeInfo.sreg = 3;
	else return OP_ERROR;

	String += 2;
	return OP_SREG;
}

eOperandType cAssembler::SolveImm(sOpCodeInfo &OpCodeInfo){
	int Value;
	if(SolveExpr(Value)){
		OpCodeInfo.imm = Value;
		if(Value < 256) return OP_IMM8;
		return OP_IMM16;
	}
	return OP_ERROR;
}

eOperandType cAssembler::SolveMemref(sOpCodeInfo &OpCodeInfo){
	const int MemRefCount = 24;
	struct sMemRefList{
		bool BX, BP, SI, DI, IMM8, IMM16;
		int mod, reg;
	} MemRefList[MemRefCount] = {
		true, false, true, false, false, false, 0, 0,
		true, false, false, true, false, false, 0, 1,
		false, true, true, false, false, false, 0, 2,
		false, true, false, true, false, false, 0, 3,
		false, false, true, false, false, false, 0, 4,
		false, false, false, true, false, false, 0, 5,
		false, false, false, false, false, true, 0, 6,
		true, false, false, false, false, false, 0, 7,

		true, false, true, false, true, false, 1, 0,
		true, false, false, true, true, false, 1, 1,
		false, true, true, false, true, false, 1, 2,
		false, true, false, true, true, false, 1, 3,
		false, false, true, false, true, false, 1, 4,
		false, false, false, true, true, false, 1, 5,
		false, true, false, false, true, false, 1, 6,
		true, false, false, false, true, false, 1, 7,

		true, false, true, false, false, true, 2, 0,
		true, false, false, true, false, true, 2, 1,
		false, true, true, false, false, true, 2, 2,
		false, true, false, true, false, true, 2, 3,
		false, false, true, false, false, true, 2, 4,
		false, false, false, true, false, true, 2, 5,
		false, true, false, false, false, true, 2, 6,
		true, false, false, false, false, true, 2, 7
	};
	bool BX, BP, SI, DI, IMM8, IMM16, IMM;
	int Value, i;
	BX = BP = SI = DI = IMM8 = IMM16 = IMM = false;
	char *Temp;

	SkipSpace();
	if(*String != '[') return OP_ERROR;
	String ++;
	for(i = 0; i < 3; i ++){
		SkipSpace();
		if(String[0] == 'b' && String[1] == 'x') BX = true;
		else if(String[0] == 'b' && String[1] == 'p') BP = true;
		else if(String[0] == 's' && String[1] == 'i') SI = true;
		else if(String[0] == 'd' && String[1] == 'i') DI = true;
		else{
			Temp = String;
			if(SolveExpr(Value)){
				IMM = true;
				break;
			}
			String = Temp;
		}
		String += 2;

		SkipSpace();
		if(*String == ']') break;
		if(*String != '+') return OP_ERROR;
		String ++;
	}
	SkipSpace();
	if(*String != ']') return OP_ERROR;
	String ++;

	if(IMM){
		if(Value < 256) IMM8 = true;
		else IMM16 = true;
	}
	for(i = 0; i < MemRefCount; i ++){
		if(MemRefList[i].BX == BX && MemRefList[i].BP == BP &&
			MemRefList[i].SI == SI && MemRefList[i].DI == DI &&
			MemRefList[i].IMM8 == IMM8 && MemRefList[i].IMM16 == IMM16){
			OpCodeInfo.mod = MemRefList[i].mod;
			OpCodeInfo.rm = MemRefList[i].reg;
			OpCodeInfo.disp = Value;
			return OP_MEM8;
		}
	}
	return OP_ERROR;
}


bool cAssembler::Solve(char *nString, int &ReturnValue){
	String = nString;

	if(SolveExpr(ReturnValue)) return true;
	
	return false;
}

bool cAssembler::SolveExpr(int &ReturnValue){
// <expr> ::= <term> + <expr> | <term> - <expr> | <term>
	int a, b;
	if(!SolveTerm(a)) return false;
	if(!SkipSpace()){
		ReturnValue = a;
		return true;
	}
	switch(*String){
	case '+':
		String ++;
		if(!SolveExpr(b)) return false;
		ReturnValue = a + b;
		return true;
	case '-':
		String ++;
		if(!SolveExpr(b)) return false;
		ReturnValue = a - b;
		return true;
	}
	ReturnValue = a;
	return true;
}

bool cAssembler::SolveTerm(int &ReturnValue){
// <term> ::= <primary> * <term> | <primary> / <term> | <primary> % <term> | <primary>
	int a, b;

	if(!SolvePrimary(a)) return false;
	if(!SkipSpace()){
		ReturnValue = a;
		return true;
	}
	switch(*String){
	case '*':
		String ++;
		if(!SolveTerm(b)) return false;
		ReturnValue = a * b;
		return true;
	case '/':
		String ++;
		if(!SolveTerm(b)) return false;
		ReturnValue = a / b;
		return true;
	case '%':
		String ++;
		if(!SolveTerm(b)) return false;
		ReturnValue = a % b;
		return true;
	}
	ReturnValue = a;
	return true;
}


bool cAssembler::SolvePrimary(int &ReturnValue){
// <primary> ::= (<expr>) | <id> | <number> 
	int i;
	// Skip Space
	if(!SkipSpace()) return false;

	if(isdigit(*String) || *String == '-'){ // Number
		if(*String == '-' || isdigit(*String) 
			&& !(String[1] == 'x' || String[1] == 'b' || String[1] == 'o') ){
			// Extract Decimal Number
			i = 0;
			if(*String == '-'){ 
				Token[i] = '-'; String ++; i ++;
			}
			while(isdigit(*String)){
				Token[i] = *String; String ++; i ++;
			}
			Token[i] = '\0';
			if(sscanf(Token, "%d", &ReturnValue) == 1) return true;
			return false;
		}
		// Another Number
		switch(*(String + 1)){
		case 'x': case 'X':
			String += 2;
			i = 0;
			while(isxdigit(*String)){
				Token[i] = *String; String ++; i ++;
			}
			Token[i] = '\0';
			if(sscanf(Token, "%x", &ReturnValue) == 1) return true;
			return false;
		}
		// Error
		return false;
	}else if(*String == '('){
		String ++;
		if(!SolveExpr(ReturnValue)) return false;
		if(!SkipSpace()) return false;
		if(*String == ')'){ 
			String ++; 
			return true;
		}
		return false;
	}else if(isalpha(*String)){
		i = 0;
		while(isalnum(*String)){
			Token[i] = *String;
			i ++;
			String ++;
		}
		Token[i] = 0;
		if(!UserSymbolTable.Get(Token, ReturnValue)){
			if(IsFirstPass){
				ReturnValue = 256;
				return true;
			}else{
				printf("Error: Symbol '%s' not found.\n", Token);
			}
			return false;
		}
		return true;

	}
	return false;
}

bool cAssembler::SkipSpace(){
	while(isspace(*String)) String ++;
	if(*String == '\0') return false;
	return true;
}

////////////////////// Extracting Line ///////////////////////////
bool cAssembler::GetIdentifier(char *Id){
	if(!SkipSpace()) return false;
	if(!isalpha(*String)) return false;

	while(isalnum(*String)){
		*Id = *String;
		String ++;
		Id ++;
	}
	*Id = '\0';
	return true;
}
////////////////////////// Extract Line ////////////////////////////
int cAssembler::ExtractLine(char *Label, char *Operator, eDirective &Directive, char *Operand){
	int LineMode = LM_NONE;
	char Buffer[256];

	*Label = *Operator = *Operand = '\0';
	Directive = D_NONE;

	if(!SkipSpace() || *String == ';') return LM_NONE;
	if(!GetIdentifier(Buffer)) return LM_ERROR;
	if(*String == ':'){
		String ++;
		strcpy(Label, Buffer);
		strlwr(Label);
		LineMode |= LM_LABEL;
		if(!SkipSpace()) return LineMode;
		if(*String == ';') return LineMode;
		if(!GetIdentifier(Buffer)) return LM_ERROR; 
	}
	Directive = GetDirectiveID(Buffer);
	if(Directive != D_NONE){
		LineMode |= LM_DIRECTIVE;
	}else{
		strcpy(Operator, Buffer);
		strlwr(Operator);
		LineMode |= LM_OPERATOR;
	}
	if(!SkipSpace()) return LineMode;
	while(*String != ';' && *String != '\n' && *String != '\0'){
		*Operand = *String;
		String ++;
		Operand ++;
	}
	*Operand = '\0';
	return LineMode;
}
