/****************************************************************************************/
/*  Engine.h                                                                            */
/*                                                                                      */
/*  Author: Charles Bloom/John Pollard                                                  */
/*  Description: Maintains the driver interface, as well as the bitmaps attached		*/
/*					to the driver.														*/
/*                                                                                      */
/*  The contents of this file are subject to the Jet3D Public License                   */
/*  Version 1.02 (the "License"); you may not use this file except in                   */
/*  compliance with the License. You may obtain a copy of the License at                */
/*  http://www.jet3d.com                                                                */
/*                                                                                      */
/*  Software distributed under the License is distributed on an "AS IS"                 */
/*  basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See                */
/*  the License for the specific language governing rights and limitations              */
/*  under the License.                                                                  */
/*                                                                                      */
/*  The Original Code is Jet3D, released December 12, 1999.                             */
/*  Copyright (C) 1996-1999 Eclipse Entertainment, L.L.C. All Rights Reserved           */
/*                                                                                      */
/****************************************************************************************/
#ifndef JE_ENGINE_H
#define JE_ENGINE_H

#include "BaseType.h"
#include "jeTypes.h"
#include "Bitmap.h"
#include "Camera.h"

#ifdef __cplusplus
extern "C" {
#endif

//-------------------------------------------------

typedef struct jeEngine			jeEngine;

typedef struct jeDriver			jeDriver;
typedef struct jeDriver_Mode	jeDriver_Mode;
typedef struct jeDriver_System	jeDriver_System;

typedef enum
{
	JE_ENGINE_BITMAP_TYPE_2D,
	JE_ENGINE_BITMAP_TYPE_3D,
} jeEngine_BitmapType;


typedef enum
{
	FrameState_None = 0,
	FrameState_Begin,
} jeEngine_FrameState;

//-------------------------------------------------
// Engine Functions
//-------------------------------------------------

// ------- Create/Destroy
#ifdef WINVER
		// include windows to get this API 
JETAPI jeEngine		*jeEngine_Create(HWND hWnd, const char *AppName, const char *DriverDirectory);
#endif

#ifdef BUILD_BE
	JETAPI jeEngine *jeEngine_Create(class G3DView* hWnd, const char *AppName, const char *DriverDirectory);
#endif
	
JETAPI jeBoolean	jeEngine_CreateRef(jeEngine *Engine);

JETAPI void			jeEngine_Free(jeEngine *Engine);			// Don't use this :
JETAPI void			jeEngine_Destroy(jeEngine **pEngine);

JETAPI jeBoolean	jeEngine_IsValid(const jeEngine *E);

// ------- Misc functions
JETAPI void			jeEngine_EnableFrameRateCounter(jeEngine *Engine, jeBoolean Enabled);

JETAPI jeBoolean	jeEngine_Activate(jeEngine *Engine, jeBoolean bActive);
JETAPI jeBoolean	jeEngine_UpdateWindow(jeEngine *Engine);

JETAPI jeBoolean	jeEngine_GetFrameState(const jeEngine *Engine, jeEngine_FrameState *FrameState);

//-------- The main Frame functions:

JETAPI jeBoolean	jeEngine_BeginFrame(jeEngine *Engine, jeCamera *Camera, jeBoolean ClearScreen);
JETAPI jeBoolean	jeEngine_EndFrame(jeEngine *Engine);

JETAPI jeBoolean	jeEngine_FlushScene(jeEngine *Engine);

// call updategamma when drivers change
JETAPI jeBoolean	jeEngine_SetGamma(jeEngine *Engine, float Gamma);
JETAPI jeBoolean	jeEngine_GetGamma(jeEngine *Engine, float *pGamma);
void					jeEngine_UpdateGamma(jeEngine *Engine);

JETAPI jeBoolean	jeEngine_AddBitmap(		jeEngine *Engine, jeBitmap *Bitmap, jeEngine_BitmapType Type);
JETAPI jeBoolean	jeEngine_RemoveBitmap(	jeEngine *Engine, jeBitmap *Bitmap);

JETAPI jeBoolean	jeEngine_ScreenShot(jeEngine *Engine, const char *FileName);

jeBoolean				jeEngine_BitmapListInit(jeEngine *Engine);
jeBoolean				jeEngine_BitmapListShutdown(jeEngine *Engine);
jeBoolean				jeEngine_AttachAll(jeEngine *Engine);
jeBoolean				jeEngine_DetachAll(jeEngine *Engine);

/*	// CB disabled in 2.0
//-------- the splash screen
jeBoolean				jeEngine_DoSplashScreen(jeEngine *Engine, jeDriver_Mode *DriverMode);
*/

//-------- engine fonts
jeBoolean				jeEngine_InitFonts(jeEngine *Engine);
jeBoolean				jeEngine_ShutdownFonts(jeEngine *Engine);
JETAPI jeBoolean	jeEngine_Printf(jeEngine *Engine, int32 x, int32 y, const char *String, ...);
jeBoolean				jeEngine_DebugPrintf(jeEngine *Engine, const char *String, ...);

// ------- Built-in drivers (parameters to jeEngine_RegisterDriver)
JETAPI	void		*jeEngine_SoftwareDriver(void);
JETAPI	void		*jeEngine_D3DDriver(void);

//-------- Engine Driver system stuff (for getting a driver to set the mode)
JETAPI jeDriver_System *jeEngine_GetDriverSystem(jeEngine *Engine);

JETAPI jeDriver		*jeDriver_SystemGetNextDriver(jeDriver_System *DriverSystem, jeDriver *Start);
JETAPI jeDriver_Mode *jeDriver_GetNextMode(jeDriver *Driver, jeDriver_Mode *Start);

JETAPI jeBoolean	jeDriver_GetName(const jeDriver *Driver, const char **Name);
JETAPI jeBoolean	jeDriver_ModeGetName(const jeDriver_Mode *Mode, const char **Name);
JETAPI jeBoolean	jeDriver_ModeGetWidthHeight(const jeDriver_Mode *Mode, int32 *pWidth, int32 *pHeight);
JETAPI jeBoolean	jeDriver_ModeGetAttributes(const jeDriver_Mode *Mode, int32 *pWidth, int32 *pHeight, int32 *pBpp);
							// CB : GetWidthHeight is deprecated ; use GetAttributes instead!

JETAPI jeBoolean 	jeEngine_RegisterDriver(jeEngine *Engine, void *HookProc);

JETAPI jeBoolean	jeEngine_GetDeviceCaps(jeEngine *pEngine, jeDeviceCaps *DeviceCaps);
JETAPI jeBoolean	jeEngine_GetDefaultRenderFlags(jeEngine *pEngine, uint32 *RenderFlags);
JETAPI jeBoolean	jeEngine_SetDefaultRenderFlags(jeEngine *pEngine, uint32 RenderFlags);

#ifdef WINVER
JETAPI jeBoolean	jeEngine_SetDriverAndMode(	jeEngine		*Engine, 
													HWND			hWnd,
													jeDriver		*Driver, 
													jeDriver_Mode	*DriverMode);
#endif

#ifdef BUILD_BE
JETAPI jeBoolean	jeEngine_SetDriverAndMode(  jeEngine		*Engine,
													class G3DView*	hWnd, // really a class G3DView *
													jeDriver		*Driver,
													jeDriver_Mode	*DriverMode);
#endif
													
JETAPI jeBoolean	jeEngine_GetDriverAndMode(	const jeEngine *Engine, 
													jeDriver **Driver, 
													jeDriver_Mode **DriverMode);

JETAPI jeBoolean	jeEngine_ShutdownDriver(jeEngine *Engine);

jeBoolean				jeEngine_ResetDriver(jeEngine *Engine);	// Resets the current driver


//--------- Render driver stuff (this is the main hard-core driver.  the engine just wraps this up with jeDriver)
//	NOTE - You MUST include dcommon.h BEFORE engine.h to use the hard-core driver functions!
#ifdef DCOMMON_H
	typedef struct jeEngine_ChangeDriverCB		jeEngine_ChangeDriverCB;

	typedef jeBoolean jeEngine_ShutdownDriverCB(DRV_Driver *Driver, void *Conext);
	typedef jeBoolean jeEngine_StartupDriverCB(DRV_Driver *Driver, void *Conext);

	// Driver will make more sense in the future (instead of RDriver)
	DRV_Driver * JETCC	jeEngine_GetDriver(const jeEngine * Engine);
	
	JETAPI jeEngine_ChangeDriverCB *jeEngine_CreateChangeDriverCB(	jeEngine					*Engine, 
																		jeEngine_ShutdownDriverCB	*ShutdownDriverCB, 
																		jeEngine_StartupDriverCB	*StartupDriverCB,
																		void						*Context);
	JETAPI void jeEngine_DestroyChangeDriverCB(jeEngine *Engine, jeEngine_ChangeDriverCB **ChangeDriverCB);
#endif

#ifdef WINVER
HINSTANCE				jeEngine_LoadLibrary( const char * lpLibFileName, const char *DriverDirectory);
#endif

//-------- drawing with the engine (Decals & Misc Polys)

JETAPI jeBoolean JETCC jeEngine_DrawBitmap(const jeEngine *Engine,
	const jeBitmap *Bitmap,	// should have been a TYPE_2D
	const jeRect * Source, uint32 x, uint32 y);

JETAPI jeBoolean JETCC jeEngine_DrawBitmap3D(const jeEngine *Engine,
	const jeBitmap *Bitmap, // should have been a TYPE_3D
	const jeRect * pRect, uint32 x, uint32 y);

JETAPI void JETCC jeEngine_RenderPoly(const jeEngine *Engine, const JE_TLVertex *Points, 
						int NumPoints, const jeBitmap *Texture, uint32 Flags);
						// Texture should have been a TYPE_3D

JETAPI void JETCC jeEngine_RenderPolyArray(const jeEngine *Engine, const JE_TLVertex ** pPoints, int * pNumPoints, int NumPolys, 
								const jeBitmap *Texture, uint32 Flags);

JETAPI void			jeEngine_FillRect(jeEngine *Engine, const jeRect *Rect, const JE_RGBA *Color);

//--------------------------------------------------------------------------

#ifdef WINVER
// Registers an Object given a handle to a DLL - Incarnadine
JETAPI jeBoolean JETCC jeEngine_RegisterObject(HINSTANCE DllHandle);

// Registers all Objects in a particular path. - Incarnadine
JETAPI jeBoolean JETCC jeEngine_RegisterObjects(char * DllPath);
#endif

#ifdef __cplusplus
}
#endif

#endif // JE_ENGINE_H
