/****************************************************************************************/
/*  FONT.H                                                                              */
/*                                                                                      */
/*  Author: Thom Robertson                                                              */
/*  Description: Bitmapped font support interface                                       */
/*               This implementation supports any TrueType fonts provided by Windows    */
/*                                                                                      */
/*  The contents of this file are subject to the Jet3D Public License                   */
/*  Version 1.02 (the "License"); you may not use this file except in                   */
/*  compliance with the License. You may obtain a copy of the License at                */
/*  http://www.jet3d.com                                                                */
/*                                                                                      */
/*  Software distributed under the License is distributed on an "AS IS"                 */
/*  basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See                */
/*  the License for the specific language governing rights and limitations              */
/*  under the License.                                                                  */
/*                                                                                      */
/*  The Original Code is Jet3D, released December 12, 1999.                             */
/*  Copyright (C) 1996-1999 Eclipse Entertainment, L.L.C. All Rights Reserved           */
/*                                                                                      */
/****************************************************************************************/
#ifndef JE_FONT_H
#define JE_FONT_H

// includes
#include "Jet.h"
#include "basetype.h"
#include "bitmap.h"
// **************************
// to use this API:

// 2.  jeFont_CreateFont().

// 3.  (Optionally) jeFont_AddCharacters().
// 3A. Otherwise, IF you intend to use jeFont_DrawText(), call jeFont_AddBitmapBuffer().

// 4.  Between jeEngine_BeginFrame() and jeEngine_EndFrame(), and after jeEngine_RenderWorld(),
//       jeFont_DrawText(). You may call jeFont_DrawTextToBitmap() anytime, though.

// 5.  When finished, jeFont_Destroy().



//*************************************************************************************** 
// these are bit flags for _DrawText(). Currently only _WORDWRAP is implemented, and without
// it, the function will still wrap, just not on word boundaries.
// Note that these will fail for non ascii fonts.
#define JE_FONT_WRAP                0x00000001  // wrap to fit inside the drawing rect
#define JE_FONT_WORDWRAP            0x00000002  // wrap on word boundaries
#define JE_FONT_JUST_RETURN_FIT     0x00000004  // returns number of characters that fit in drawing rectangle, WITHOUT drawing anything.
#define JE_FONT_JUSTIFY_RIGHT       0x00000008
#define JE_FONT_JUSTIFY_CENTER      0x00000010

#ifdef __cplusplus
extern "C" {
#endif

// opaque structure headers.
typedef struct jeFont jeFont;			// an instance of a font


//*************************************************************************************** 
JETAPI jeFont *JETCC jeFont_Create(const jeEngine *Engine, const char *fontNameString, 
                                               const int fontSize,
											   const int fontWeight , const jeBoolean antialiased) ;
   // Creates a font, and returns a pointer to it.
   // Pass in the string name of the TrueType font (case sensitive), and the height in pixels.
   
   // ARGUMENTS:
   // fontNameString - char pointer to a string containing the case sensitive name of the font.
   // fontSize - the pixel height of the requested font.

   // RETURNS:
   // success: pointer to the newly created font.
   // failure: NULL.

   // NOTE: the new font set has NO actual characters in it at first.  You must add characters
   // to it with the _AddCharacters() function before you can use the font.
   // NOTE: all fonts start out with a grayscale palette, with the range 0 to 128.

//*************************************************************************************** 
JETAPI void JETCC jeFont_CreateRef(jeFont *font);


//*************************************************************************************** 
JETAPI void JETCC jeFont_Destroy(jeFont **font);
   // destroys a font.

   // ARGUMENTS:
   // font - pointer to the font to be destroyed.

   // RETURNS:
   // nothing.

//*************************************************************************************** 
JETAPI jeBoolean JETCC jeFont_AddCharacters(jeFont *font, 
                                                  unsigned char leastIndex, 
                                                  unsigned char mostIndex
                                                  );
   // Adds a set of characters to the font defined by the ascii range passed in 
   // (leastIndex and mostIndex, inclusive).

   // ARGUMENTS:
   // font - pointer to the font to add characters to.
   // e - pointer to a valid jeEngine.
   // leastIndex and mostIndex - the ASCII range of characters to add.
   // cellBuffer - an allocated hunk of ram to temproarily store the character image
   // bufferSize - length of the above buffer

   // RETURNS:
   // success: JE_TRUE.
   // failure: JE_FALSE.

   // NOTES:
   // This is the function that actually uses the
   // Win32 GetGlyphOutline() function to draw the character onto a jeBitmap, which can be
   // blitted to the screen.


//*******************************************************************************
JETAPI void JETCC jeFont_DestroyBitmapBuffer( jeFont *font );
   // destroys any valid "scratch-pad" buffer attached to the jeFont.
   // ARGUMENTS:
   // font - pointer to the jeFont.
   //
   // NOTES:
   // you'll rarely need to call this function; it's called by jeFont_Destroy() anyway.
   // Calling this function with a jeFont that has no initialized buffer doesn't
   // hurt anything.

//*******************************************************************************
JETAPI jeBoolean JETCC jeFont_AddBitmapBuffer(
                                  jeFont *font, const uint32 width, const uint32 height);
   // Adds a jeBitmap to the jeFont, to be used as a temporary "scratch-pad".  This is
   // required for using jeFont_DrawText() when no characters have been added.

   // ARGUMENTS:
   // font - pointer to the jeFont to add a buffer to.
   // width and height - the size of the buffer to create.  Make sure this size is >=
   // the biggest rectagle of text you'll want to write to the screen using this jeFont
   // and DrawText().

   // RETURNS:
   // success: JE_TRUE.
   // failure: JE_FALSE.

   // NOTES:
   // You don't need to call this function IF you _AddCharacters() to this jeFont.
   // You call this function for each jeFont you need to use.  jeFont's don't share buffers.
   // if you call this function on a jeFont that already has a valid buffer, the buffer is
   // destroyed, and replaced by the new one.

//*************************************************************************************** 
JETAPI jeBoolean JETCC jeFont_DrawText(jeFont *font, const char *textString, 
                                           const JE_Rect *Rect, const JE_RGBA *Color, 
                                           uint32 flags, const JE_Rect *clipRect);
   // This is the function you put between jeEngine_BeginFrame() and _EndFrame(), the function
   // that draws text to the screen.

   // ARGUMENTS:
   // font - pointer to the font to draw with.  IF the font has NO characters in it
   //  (added by jeFont_AddCharacters() ) then a different, more windows-intensive way is
   //  used to draw out the characters.
   // textString - pointer to the text string to output to the screen.
   // Rect - screen rectangle to place the text within.
   // Color - RGB color the text should be.
   // flags - a bitfield of JE_FONT_ values.
   // clipRect - pointer to a screen rectangle to clip the text to.  MAY BE NULL, in which
   // case the text is only clipped by the boundaries of the screen.

   // RETURNS:
   // success: JE_TRUE.
   // failure: JE_FALSE.

   // NOTES:
   // Assuming you've added characters to the font, characters which have NOT been added
   // WILL cause an assert if you try to draw them.  
   // Only JE_FONTSET_WORDWRAP is meaningfull right now.  Using any other flags will cause
   // an assert.
   // As stated above, you can use an entirely different way of creating a string, by
   // making a font with no characters in it.  This
   // jumps through Windows DIB hoops, and draws the text in a non-anti-aliased, but
   // (hopefully) more unicode-tolerant way (DrawText() ).


//*************************************************************************************** 
JETAPI jeBoolean JETCC jeFont_DrawTextToBitmap(jeFont *font, const char *textString, 
                                           const JE_Rect *Rect, const JE_RGBA *Color, 
                                           uint32 flags, const JE_Rect *clipRect,
                                           jeBitmap *targetBitmap);
   // This is the function you put between jeEngine_BeginFrame() and _EndFrame(), the function
   // that draws text to the screen.

   // ARGUMENTS:
   // font - pointer to the font to draw with.  IF the font has NO characters in it
   //  (added by jeFont_AddCharacters() ) then a different, more windows-intensive way is
   //  used to draw out the characters.
   // textString - pointer to the text string to output to the screen.
   // Rect - screen rectangle to place the text within.
   // Color - RGB color the text should be.
   // flags - a bitfield of JE_FONT_ values.
   // clipRect - pointer to a screen rectangle to clip the text to.  MAY BE NULL, in which
   // case the text is only clipped by the boundaries of the screen.
   // targetBitmap - pointer to a target bitmap to draw the text into.  MAY NOT BE NULL,
   // and MUST BE JE_PIXELFORMAT_8BIT.

   // RETURNS:
   // success: JE_TRUE.
   // failure: JE_FALSE.

   // NOTES:
   // Assuming you've added characters to the font, characters which have NOT been added
   // WILL cause an assert if you try to draw them.  
   // Only JE_FONTSET_WORDWRAP is meaningfull right now.  Using any other flags will cause
   // an assert.
   // As stated above, you can use an entirely different way of creating a string, by
   // making a font with no characters in it.  This
   // jumps through Windows DIB hoops, and draws the text in a non-anti-aliased, but
   // (hopefully) more unicode-tolerant way (DrawText() ).
   // The Color argument is will be used to modify the existing palette of the targetBitmap
   // passed in.  Therefore, you won't be able to _DrawTextToBitmap() a red piece of text,
   // then a green piece, then a blue piece.  You'll end up with three lines of blue text.


//*************************************************************************************** 
JETAPI int32 JETCC jeFont_GetStringPixelWidth (jeFont *font, const char *textString);
JETAPI int32 JETCC jeFont_GetStringPixelHeight(jeFont *font, const char *textString);
   // These two functions return the pixel width and height of the string passed in.

   // ARGUMENTS:
   // font - pointer to the font to draw with.
   // textString - pointer to the text string to output to the screen.

   // RETURNS:
   // success: a positive value in pixels.  IF the text passed in contains characters
   //          which haven't been added to the font yet, BUT other characters HAVE
   //          been added, the function asserts.
   // failure: -1.
   // NOTES:
   // these two functions assume no text wrapping!

//*************************************************************************************** 
JETAPI jeBitmap* JETCC jeFont_GetBuffer(jeFont *font);
   // This function returns a pointer to the drawing buffer contained by the font.

   // ARGUMENTS:
   // font - pointer to the font.

   // RETURNS:
   // a valid pointer to a jeBitmap, OR NULL, signifying that the buffer wasn't initialized.


//*************************************************************************************** 
JETAPI jeBoolean JETCC jeFont_GetCharMap(jeFont *font, uint8 character, JE_Rect *Rect, 
												 jeBitmap **targetBitmap, int32 *fullWidth, int32 *fullHeight, 
												 int32 *offsetX, int32 *offsetY);



//*************************************************************************************** 
JETAPI void JETCC jeFont_EnableAntialiasing(jeFont *font, const jeBoolean anti);
//*************************************************************************************** 
JETAPI jeBoolean JETCC jeFont_IsAntialiased(jeFont *font);

#ifdef __cplusplus
}
#endif

#endif
