/****************************************************************************************/
/*  SHADER_MAP.C                                                                        */
/*                                                                                      */
/*  Author: Timothy Roff	                                                            */
/*  Description:  Shader implementation                                                 */
/*                                                                                      */
/*  The contents of this file are subject to the Jet3D Public License                   */
/*  Version 1.02 (the "License"); you may not use this file except in                   */
/*  compliance with the License. You may obtain a copy of the License at                */
/*  http://www.jet3d.com                                                                */
/*                                                                                      */
/*  Software distributed under the License is distributed on an "AS IS"                 */
/*  basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See                */
/*  the License for the specific language governing rights and limitations              */
/*  under the License.                                                                  */
/*                                                                                      */
/*  This file was not part of the original Jet3D, released December 12, 1999.           */
/*                                                                                      */
/****************************************************************************************/

#include "jeShaderDefs.h"
#include "jeShader_h.h"

typedef enum {
   MAP_TYPE_NORMAL,
   MAP_TYPE_SOLID_SKY,
   MAP_TYPE_ALPHA_SKY,

};

jeBoolean request_map(sh_stage_t stage, const char* name, int m, jeBoolean clampmap, int type)
{
   // assert (m < JE_SHADER_MAX_ANIMAP)

   jeShader_Error("DONT CALL REQUEST_MAP!!!! ERROR!!!");

	if (strlen(name) > JE_SHADER_NAME_LEN) {
      jeShader_Error("map name '%s' exceeds JE_SHADER_NAME_LEN (%d) characters", name, JE_SHADER_NAME_LEN);
      return JE_FALSE;
   }
   else {
      strcpy(stage.maps[m].name, name);
	  //stage.maps[m].name holds the texture info!!!
	     
      if (clampmap) stage.maps[m].flags |= SH_MAP_CLAMPMAP;

      if (type == MAP_TYPE_SOLID_SKY) {
         stage.maps[m].flags |= SH_MAP_SOLID_SKY_EXT;
      }
      else if (type == MAP_TYPE_ALPHA_SKY) {
         stage.maps[m].flags |= SH_MAP_ALPHA_SKY_EXT;
      }

      return JE_TRUE;
   }
}


//=====================================================================================
//	mapParseFunc
//=====================================================================================
jeBoolean mapParseFunc(int unused, char* argv[], shader_info_t *info)
{
	
	jeBoolean is_clampmap;

	assert(info != NULL);

	#ifdef _DEBUG
	jeShader_Error("in map");
	#endif

	(void)unused;

	is_clampmap = (stricmp(argv[0], "clampmap") == 0);

	if (is_clampmap)
	{
		if (stricmp(argv[1], "$lightmap") == 0)
		{
			jeShader_Error("$lightmap cannot be a clampmap");
			return JE_TRUE;
		}

		if (stricmp(argv[1], "$whiteimage") == 0)
		{
			jeShader_Error("$whiteimage cannot be a clampmap");
			return JE_TRUE;
		}
	}
	sprintf(info->shader_vars.texture_maps[0], argv[1]);

	
	info->stage.anim_map_count = 1;
	info->shader_vars.used_maps = info->stage.anim_map_count;
	
	return JE_TRUE;
}

//=====================================================================================
//	animMapParseFunc
//=====================================================================================
jeBoolean animMapParseFunc(int argc, char* argv[], shader_info_t *info)
{
	int i;
	int anim_map_count;

	assert(info != NULL);

#ifdef _DEBUG
	jeShader_Error("entering animMapParseFunc!!!");
#endif
	
	shader_number_parse(argv[1], &info->stage.anim_map_fps);

   anim_map_count = 0;

   for (i = 2; i < argc; i++) {
      if (stricmp(argv[1], "$lightmap") == 0) {
         jeShader_Error("$lightmap cannot be an animMap");
         continue;
      }

      if (stricmp(argv[1], "$whiteimage") == 0) {
         jeShader_Error("$whiteimage cannot be an animMap");
         continue;
      }

      /*if (request_map(info->stage, argv[i], anim_map_count, JE_FALSE, MAP_TYPE_NORMAL)) {
	  {
         anim_map_count++;
      }*/
   }

   if (anim_map_count > 0) {
      if (anim_map_count > 8) {
         jeShader_Error("having more than 8 maps is an extension");
      }

      if (anim_map_count > JE_SHADER_MAX_ANIMAP) {
         jeShader_Error("number of animMap frames exceeds JE_SHADER_MAX_ANIMAP (%d) in info->shader '%s'", JE_SHADER_MAX_ANIMAP, info->shader->name);
      }

      info->stage.anim_map_count = anim_map_count;
   }

   return JE_TRUE;
}

//=====================================================================================
//	jeSkyMapParseFunc
//=====================================================================================
jeBoolean jeSkyMapParseFunc(int unused, char* argv[], shader_info_t *info)
{
	jeBoolean is_alpha_map;

	assert(info != NULL);

   (void)unused;

   jeShader_Error("jeSkyMap is an extension");

   if (strnicmp(argv[2], "$bsp#sky", 8) != 0) {
      jeShader_Error("jeSkyMap only works with sky maps from BSP files");
      return JE_TRUE;
   }

   if (stricmp(argv[1], "alpha") == 0) {
      is_alpha_map = JE_TRUE;
   }
   else if (stricmp(argv[1], "solid") == 0) {
      is_alpha_map = JE_FALSE;
   }
   else {
      jeShader_Error("type must by solid or alpha");
      return JE_TRUE;
   }

   /*if (request_map(info->stage, argv[2], 0, JE_FALSE, is_alpha_map?MAP_TYPE_ALPHA_SKY:MAP_TYPE_SOLID_SKY)) {
      info->stage.anim_map_count = 1;
   }*/

   return JE_TRUE;
}

//=====================================================================================
//	ParseSkyboxBasename
//=====================================================================================
jeBoolean ParseSkyboxBasename(const char* in, char* out, const char* skybox)
{
   if (strcmp("-", in) != 0) {
      if (strlen(in) <= JE_SHADER_NAME_LEN - sizeof("_xx.bmp")) {
         strcpy(out, in);
         return JE_TRUE;
      }
      else {
         jeErrorLog_AddString(JE_ERR_SHADER_SCRIPT, "base name too long: ", skybox);
      }
   }

   out[0] = '\0';

   return JE_FALSE;
}

//=====================================================================================
//	skyParmsParseFunc
//=====================================================================================
jeBoolean skyParmsParseFunc(int unused, char* argv[], shader_info_t *info)
{
	assert(info != NULL);

   (void)unused;

   // farbox
   if (ParseSkyboxBasename(argv[1], info->shader->far_box_basename, "far box")) {
      info->shader->flags |= SH_FAR_BOX;
   }
   else {
      info->shader->flags &= ~SH_FAR_BOX;
   }

   // cloudheight
   if (strcmp("-", argv[2]) != 0) {
      shader_number_parse(argv[2], &info->shader->cloud_height);
      info->shader->flags |= SH_EXPLICIT_SKYGEN;
   }
   else {
      shader_number_parse("128", &info->shader->cloud_height);
   }

   info->shader->flags |= SH_SKYGEN;

   // nearbox
   if (ParseSkyboxBasename(argv[3], info->shader->near_box_basename, "near box")) {
      info->shader->flags |= SH_NEAR_BOX;
   }
   else {
      info->shader->flags &= ~SH_NEAR_BOX;
   }

   return JE_TRUE;
}