// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for Common C++, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#ifndef	__CCXX_HTML_H__
#define	__CCXX_MTML_H__

#ifndef	__CCXX_PERSIST_H__
#include <cc++/persist.h>
#endif

#include <vector>
#include <map>
#include <string>

  /**
   * Writer
   *
   * This class must be overridden if you need anything other than Writing
   * to stdout
   *
   * @author Daniel Silverstone
   * @short output html to stdout.
   */
  class Writer
  {
  public:
	virtual void WriteSomeOutput(const string& stuff, bool newLine);
  };

  /**
   * Entity
   * 
   * This holds the basic state for all HTML entries.
   * It is capable of being serialised and can hold arbitrary information.
   * 
   * Most of the methods are protected, suggesting that they should only
   * be used by subclasses and by "promotion" to public
   *
   * @author Daniel Silverstone
   * @short state entity for all HTML entries.
   */
  class Entity : public BaseObject
  {
	DECLARE_PERSISTENCE(Entity)
  public:
	Entity();
	/**
	 * Special case of name=="" - no tags are emitted, 
	 *                            attrs entities and settings ignored only
	 *                            content is streamed to html.
	 */
	Entity(const string& name);
	virtual ~Entity();

	/**
	 * These are the important functions which need to be overridden properly for persistence.
	 * Failure to do so results in Persistence::Engine barfing
	 */
	virtual bool Write(Engine& archive) const;
	virtual bool Read(Engine& archive);
	
	/**
	 * This function need to be overridden for streaming as HTML
	 *
	 * HTML form is <name attributes settings>[content][entities]{ifterminated </tag>}
	 */
	virtual void OutputHTML(Writer& wr) const;

  protected:
	// Typedefs
	typedef vector<string> Strings;
	typedef map<string,string> StringMap;
	typedef vector<Entity*> Entities;

	/**
	 * This returns the Attribute list
	 * They get dumped in the tag as entities with no '='
	 */
	Strings & GetAttributes() { return myAttributes; }

	/**
	 * This returns the Settings Map
	 * They get dumped in the tag as entities with a '=value'
	 */
	StringMap & GetSettings() { return mySettings; }

	/**
	 * This returns the content reference
	 */
	string & GetContent() { return myContent; }

	/**
	 * These get/set whether or not the tag has an </tag> or not
	 */
	void SetWhetherTerminated(bool yesno) { myTerminated = yesno; }
	bool GetWhetherTerminated() const { return myTerminated; }

	/**
	 * These are the Children of the Entity.
	 * They automatically get Serialised
	 */
	Entities & GetEntities() { return myEntities; }

	/**
	 * This get/set 's the newline after tag setting.
	 * This is not propogated to subentities
	 */
	void SetNewlineAfterTag(bool yesno) { myNewLine = yesno; }
	bool GetNewlineAfterTag() const { return myNewLine; }

  private:
	string myName;
	string myContent;
	Strings myAttributes;
	StringMap mySettings;
	bool myTerminated;
	bool myNewLine;
	Entities myEntities;
  };

  // Forward declarations...

  class Head;
  class Body;

  /**
   * HTMLDocument
   *
   * This class embodies the concept of the HTML Document.
   * It encapsulates the <HTML></HTML> tags and allows only
   * Two sub-Entities - namely a Head and a Body which it
   * automatically contains for you.
   *
   * @author Daniel Silverstone
   * @short whole html document entity.
   */
  class HTMLDocument : public Entity
  {
	DECLARE_PERSISTENCE(HTMLDocument)
  public:
	HTMLDocument();
	virtual ~HTMLDocument();

	HTMLDocument(bool frameset);
	/**
	 * Return the Head entity for the document.
	 *
	 * This entity deals with things like <TITLE> etc.
	 */
	Head& GetHeadEntity();
	/**
	 * Return the Body entity for the document.
	 *
	 * This entity deals with all the information (body) of the document
	 */
	Body& GetBodyEntity();

	/**
	 * This is overridden to allow us to write the DTD before we enter the <HTML> tag
	 */
	virtual void OutputHTML(Writer& wr) const;
  };

  /**
   * The Head Entity encapsulates the header of the HTML document.
   *
   * It contains entities such a <TITLE> and similar.
   *
   * @author Daniel Silverstone
   * @short HTML entity for document header.
   */
  class Head : public Entity
  {
	DECLARE_PERSISTENCE(Head)
  public:
	Head();
	~Head();
  };

  /**
   * The Body entity encapsulates the HTML document's text.
   *
   * @author Daniel Silverstone
   * @short HTML entity for document body.
   */
  class Body : public Entity
  {
	DECLARE_PERSISTENCE(Body)
  public:
	Body();
	~Body();
  };

  /**
   * The Frameset entity is a specialised Body.
   * 
   * If you try to add entities to a frameset they go into the
   * <NOFRAMES> subentity unless you add another frameset or
   * a frame
   *
   * @author Daniel Silverstone
   * @short HTML entity for doucment frameset container.
   */
  class Frameset : public Body
  {
	DECLARE_PERSISTENCE(Frameset)
  public:
	Frameset();
	~Frameset();
  };

#endif

/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
