// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
//
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
//
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for Common C++, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "config.h"
#include "macros.h"
#include "thread.h"
#include "socket.h"
#include <errno.h>


TimerPort::TimerPort()
{
	active = false;
	gettimeofday(&timer, NULL);
}

void TimerPort::setTimer(timeout_t timeout)
{
	gettimeofday(&timer, NULL);
	active = false;
	if(timeout)
		incTimer(timeout);
}

void TimerPort::incTimer(timeout_t timeout)
{
	int secs = timeout / 1000;
	int usecs = (timeout % 1000) * 1000;

	timer.tv_usec += usecs;
	if(timer.tv_usec > 1000000l)
	{
		++timer.tv_sec;
		timer.tv_usec %= 1000000l;
	}
	timer.tv_sec += secs;
	active = true;
}

void TimerPort::endTimer(void)
{
	active = false;
}

timeout_t TimerPort::getTimer(void)
{
	struct timeval now;
	long diff;

	if(!active)
		return TIMEOUT_INF;

	gettimeofday(&now, NULL);
	diff = (timer.tv_sec - now.tv_sec) * 1000l;
	diff += (timer.tv_usec - now.tv_usec) / 1000l;

	if(diff < 0)
		return 0l;

	return diff;
}

SocketPort::SocketPort(SocketService *svc, TCPSocket &tcp) :
	Socket(accept(tcp.so, NULL, NULL)),
	detect_pending(true),
	detect_output(false),
	detect_disconnect(true)
{
	next = prev = NULL;
	service = NULL;

	if(so > -1)
	{
		setError(false);
		service = svc;
		svc->Attach(this);
	}
}

SocketPort::SocketPort(SocketService *svc, const InetAddress &ia, 
		       tpport_t port) :
Socket(AF_INET, SOCK_DGRAM, 0)
{
	struct sockaddr_in addr;

	next = prev = NULL;
	service = NULL;
	addr.sin_family = AF_INET;
	addr.sin_addr = getaddress(ia);
	addr.sin_port = htons(port);
	if(bind(so, (struct sockaddr *)&addr, (socklen_t)sizeof(addr)))
	{
		endSocket();
		Error(SOCKET_BINDING_FAILED);
		return;
	}
	state = SOCKET_BOUND;
	service = svc;
	setError(false);
	svc->Attach(this);
}

SocketPort::~SocketPort()
{
	if(service) {
		service->Detach(this);
	}
	endSocket();
}

sockerror_t SocketPort::Connect(const InetAddress &ia, tpport_t port)
{
	struct sockaddr_in addr;
	long opts;
	sockerror_t rtn = SOCKET_SUCCESS;

	addr.sin_family = AF_INET;
	addr.sin_addr = getaddress(ia);
	addr.sin_port = htons(port);

	opts = fcntl(so, F_GETFL);
	fcntl(so, F_SETFL, opts | O_NDELAY);
	if(connect(so, (struct sockaddr *)&addr, (socklen_t)sizeof(addr)))
		rtn = connectError();
	fcntl(so, F_SETFL, opts);
	return rtn;
}


void SocketPort::setTimer(timeout_t ptimer)
{
	TimerPort::setTimer(ptimer);
	service->Update();
}

void SocketPort::incTimer(timeout_t ptimer)
{
	TimerPort::incTimer(ptimer);
	service->Update();
}

void SocketPort::setDetectPending( bool val )
{
	if ( detect_pending != val ) {
		detect_pending = val;
#ifdef __CCXX_USE_POLL
		if ( ufd ) {
			if ( val ) {
				ufd->events |= POLLIN;
			} else {
				ufd->events &= ~POLLIN;
			}
		}
#endif
		service->Update();
	}
}

void SocketPort::setDetectOutput( bool val )
{
	if ( detect_output != val ) {
		detect_output = val;
#ifdef __CCXX_USE_POLL
		if ( ufd ) {
			if ( val ) {
				ufd->events |= POLLOUT;
			} else {
				ufd->events &= ~POLLOUT;
			}
		}
#endif
		service->Update();
	}
}

SocketService::SocketService(int pri) :
Thread(NULL, pri), Mutex()
{
	long opt;

	first = last = NULL;
	count = 0;
	FD_ZERO(&connect);
	::pipe(iosync);
	hiwater = iosync[0] + 1;
	FD_SET(iosync[0], &connect);

	opt = fcntl(iosync[0], F_GETFL);
	fcntl(iosync[0], F_SETFL, opt | O_NDELAY);	
}

SocketService::~SocketService()
{

	Update(0);
	Terminate();

	while(first)
		delete first;
}

void SocketService::Attach(SocketPort *port)
{
	EnterMutex();
#ifdef	__CCXX_USE_POLL
	port->ufd = 0;
#endif
	if(last)
		last->next = port;

	port->prev = last;
	last = port;
	FD_SET(port->so, &connect);
	if(port->so >= hiwater)
		hiwater = port->so + 1;

	if(!first)
	{
		first = port;
		LeaveMutex();
		Start();
	}
	else
	{
		LeaveMutex();
		Update();
	}			
	++count;
}

void SocketService::Detach(SocketPort *port)
{
	EnterMutex();
#ifndef __CCXX_USE_POLL
	FD_CLR(port->so, &connect);
#endif
	if(port->prev) {
		port->prev->next = port->next;
	} else {
		first = port->next;
	}
	
	if(port->next) {
		port->next->prev = port->prev;
	} else {
		last = port->prev;
	}
	
	--count;
	LeaveMutex();
	Update();
}



void SocketService::Update(unsigned char flag)
{
	::write(iosync[1], (char *)&flag, 1);
}


void SocketService::Run(void)
{
	timeout_t timer, expires;
	SocketPort *port;
	unsigned char buf;
	int so;

#ifdef 	__CCXX_USE_POLL
	
	Poller			  mfd;
	pollfd			* p_ufd;
	int				  lastcount = 0;

	// initialize ufd in all attached ports :
	// probably don't need this but it can't hurt.
	EnterMutex();
	port = first;
	while(port)
	{
		port->ufd = 0;
		port = port->next;
	}
	LeaveMutex();
	
#else
	struct timeval timeout, *tvp;
	fd_set inp, out, err;
	FD_ZERO(&inp);
	FD_ZERO(&out);
	FD_ZERO(&err);
#endif

	setCancel(THREAD_CANCEL_DEFERRED);
	for(;;)
	{
		timer = TIMEOUT_INF;
		while(1 == ::read(iosync[0], (char *)&buf, 1))
		{
			if(buf)
			{
				OnUpdate(buf);
				continue;
			}

			setCancel(THREAD_CANCEL_IMMEDIATE);
			Sleep(TIMEOUT_INF);
			Exit();
		}

#ifdef	__CCXX_USE_POLL

		bool	reallocate = false;
		
		EnterMutex();
		OnEvent();
		port = first;
		while(port)
		{
			OnCallback(port);
			if ( ( p_ufd = port->ufd ) ) {

				if ( ( POLLHUP | POLLNVAL ) & p_ufd->revents ) {
					// Avoid infinite loop from disconnected sockets
					port->detect_disconnect = false;
					p_ufd->events &= ~POLLHUP;

					SocketPort* p = port;
					port = port->next;
					p->Disconnect();
					continue;
				}
	
				if ( ( POLLIN | POLLPRI ) & p_ufd->revents )
					port->Pending();
	
				if ( POLLOUT & p_ufd->revents )
					port->Output();

			} else {
				reallocate = true;
			}

retry:
			expires = port->getTimer();

			if(expires > 0)
				if(expires < timer)
					timer = expires;

			if(!expires)
			{
				port->endTimer();
				port->Expired();
				goto retry;
			}

			port = port->next;
		}

		//
		// reallocate things if we saw a ServerPort without
		// ufd set !
		if ( reallocate || ( ( count + 1 ) != lastcount ) ) {
			lastcount = count + 1;
			p_ufd = mfd.getList( count + 1 );
	
			// Set up iosync polling
			p_ufd->fd = iosync[0];
			p_ufd->events = POLLIN | POLLHUP;
			p_ufd ++;
			
			port = first;
			while(port)
			{
				p_ufd->fd = port->so;
				p_ufd->events =
					( port->detect_disconnect ? POLLHUP : 0 )
					| ( port->detect_output ? POLLOUT : 0 )
					| ( port->detect_pending ? POLLIN : 0 )
				;
				port->ufd = p_ufd;
				p_ufd ++;
				port = port->next;
			}
		}
		LeaveMutex();

		poll( mfd.getList(), count + 1, timer );

#else
		EnterMutex();
		OnEvent();
		port = first;
		while(port)
		{
			OnCallback(port);
			so = port->so;
			if(FD_ISSET(so, &err)) {
				port->detect_disconnect = false;
				
				SocketPort* p = port;
				port = port->next;
				p->Disconnect();
				continue;
			}

			if(FD_ISSET(so, &inp))
				port->Pending();

			if(FD_ISSET(so, &out))
				port->Output();

retry:
			expires = port->getTimer();
			if(expires > 0)
				if(expires < timer)
					timer = expires;

			// if we expire, get new scheduling now

			if(!expires)
			{
				port->endTimer();
				port->Expired();
				goto retry;
			}

			port = port->next;
		}

		FD_ZERO(&inp);
		FD_ZERO(&out);
		FD_ZERO(&err);
		port = first;
		while(port)
		{

			if(port->detect_pending)
				FD_SET(so, &inp);

			if(port->detect_output)
				FD_SET(so, &out);

			if(port->detect_disconnect)
				FD_SET(so, &err);

			port = port->next;
		}
		
		LeaveMutex();
		if(timer == TIMEOUT_INF)
			tvp = NULL;
		else
		{
			tvp = &timeout;
			timeout.tv_sec = timer / 1000;
			timeout.tv_usec = (timer % 1000) * 1000;
		}
		select(hiwater, &inp, &out, &err, tvp);		
#endif
	}
}						
		
	
/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
