// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for Common C++, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "config.h"
#include "macros.h"
#include "thread.h"
#include "slog.h"
#include <syslog.h>

Slog slog;

Slog::Slog() :
streambuf(), ostream()
{
	ostream::init((streambuf *)this);
	_enable = true;
	_level = SLOG_DEBUG;
}

Slog::~Slog()
{
	closelog();
}

void Slog::close(void)
{
	closelog();
}

void Slog::open(const char *ident, slog_class_t grp)
{
	int fac;
	switch(grp)
	{
	case SLOG_USER:
		fac = LOG_USER;
		break;

	case SLOG_DAEMON:
		fac = LOG_DAEMON;
		break;

	case SLOG_AUDIT:
#ifdef	LOG_AUTHPRIV
		fac = LOG_AUTHPRIV;
		break;
#endif
	case SLOG_SECURITY:
		fac = LOG_AUTH;
		break;

	default:
		fac = LOG_USER;
		break;
	}
	openlog(ident, 0, fac);
}

int Slog::overflow(int c)
{
	Thread *thread = getThread();

	if(c == '\n' || !c)
	{
		if(!thread->_msgpos)
			return c;

		thread->_msgbuf[thread->_msgpos] = 0;
		if(_enable)
			syslog(priority, "%s", thread->_msgbuf);
		thread->_msgpos = 0;
		if(getppid() > 1 && _enable)
			clog << thread->_msgbuf << endl;
		_enable = true;
		return c;
	}

	if(thread->_msgpos < sizeof(thread->_msgbuf) - 1)
		thread->_msgbuf[thread->_msgpos++] = c;

	return c;
}

Slog &Slog::operator()(const char *ident, slog_class_t grp, slog_level_t
level)
{
	Thread *thread = getThread();

	thread->_msgpos = 0;
	_enable = true;
	open(ident, grp);
	return this->operator()(level, grp);
}

Slog &Slog::operator()(slog_level_t level, slog_class_t grp)
{
	Thread *thread = getThread();

	thread->_msgpos = 0;
	if(_level >= level)
		_enable = true;
	else
		_enable = false;

	switch(level)
	{
	case SLOG_EMERGENCY:
		priority = LOG_EMERG;
		break;
	case SLOG_ALERT:
		priority = LOG_ALERT;
		break;
	case SLOG_CRITICAL:
		priority = LOG_CRIT;
		break;
	case SLOG_ERROR:
		priority = LOG_ERR;
		break;
	case SLOG_WARNING:
		priority = LOG_WARNING;
		break;
	case SLOG_NOTICE:
		priority = LOG_NOTICE;
		break;
	case SLOG_INFO:
		priority = LOG_INFO;
		break;
	case SLOG_DEBUG:
		priority = LOG_DEBUG;
		break;
	}
	switch(grp)
	{
	case SLOG_AUDIT:
#ifdef	LOG_AUTHPRIV
		priority |= LOG_AUTHPRIV;
		break;
#endif
	case SLOG_SECURITY:
		priority |= LOG_AUTH;
		break;
	case SLOG_USER:
		priority |= LOG_USER;
		break;
	case SLOG_DAEMON:
		priority |= LOG_DAEMON;
		break;
	case SLOG_DEFAULT:
		priority |= LOG_USER;
		break;
	}
	return *this;
}

Slog &Slog::operator()(void)
{
	return *this;
}	
/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
